<?php
use Illuminate\Support\Facades\Auth;

use App\User;
use App\Client;
use App\Spsinfo;
use App\Transactionlog;
use App\Paymentlog;
use App\Transferlog;
use App\Transfererrorlog;
use App\Transfermanager;
use App\Billet;
use App\Billeterror;
use App\Smartsafetransfer;
use App\Loginaccesslog;
use App\Usedavailablelimit;
use App\CollectsMap;
use App\PaidUsedLimits;

use App\Darfpayment;
use App\Darjpayment;
use App\Fgtspayment;
use App\Gpspayment;
use App\Garepayment;
use App\Barcodepayment;
use App\Bank;
use App\Classes\ClientSPS\ClientSpsService;
use App\Spsbank;
use App\Multisafe;

use App\Classes\Log;
use Faker\Provider\Lorem;
use Illuminate\Support\Facades\Storage;
use phpDocumentor\Reflection\Types\Boolean;

function replaceNullValueWithEmptyString($value) {
    $value = is_null($value)? "" : $value;
    return $value;
}

function formataCNPJ($number) {
    $string = preg_replace('/[^0-9]/s', "", $number);

    if (strlen($number) == 14)
        $string = substr($string, 0, 2) . '.' . substr($string, 2, 3) . '.' . substr($string, 5, 3) . '/' . substr($string, 8, 4) . '-' . substr($string, 12, 2);

    if (strlen($number) == 11)
        $string = substr($string, 0, 3) . '.' . substr($string, 3, 3) . '.' . substr($string, 6, 3) . '-' . substr($string, 9, 2);

    return $string;
}

function validaCNPJ($cnpj)
{
    $cnpj = preg_replace('/[^0-9]/', '', (string) $cnpj);

    // Valida tamanho
    if (strlen($cnpj) != 14)
        return false;

    // Verifica se todos os digitos são iguais
    if (preg_match('/(\d)\1{13}/', $cnpj))
        return false;

    // Valida primeiro dígito verificador
    for ($i = 0, $j = 5, $soma = 0; $i < 12; $i++)
    {
        $soma += $cnpj[$i] * $j;
        $j = ($j == 2) ? 9 : $j - 1;
    }

    $resto = $soma % 11;

    if ($cnpj[12] != ($resto < 2 ? 0 : 11 - $resto))
        return false;

    // Valida segundo dígito verificador
    for ($i = 0, $j = 6, $soma = 0; $i < 13; $i++)
    {
        $soma += $cnpj[$i] * $j;
        $j = ($j == 2) ? 9 : $j - 1;
    }

    $resto = $soma % 11;

    return $cnpj[13] == ($resto < 2 ? 0 : 11 - $resto);
}

function somenteNumeros($num) {
    return preg_replace( '/[^0-9]/is', '', $num );
}

function validaCPF($cpf) {

    // Extrai somente os números
    $cpf = preg_replace( '/[^0-9]/is', '', $cpf );

    // Verifica se foi informado todos os digitos corretamente
    if (strlen($cpf) != 11) {
        return false;
    }

    // Verifica se foi informada uma sequência de digitos repetidos. Ex: 111.111.111-11
    if (preg_match('/(\d)\1{10}/', $cpf)) {
        return false;
    }

    // Faz o calculo para validar o CPF
    for ($t = 9; $t < 11; $t++) {
        for ($d = 0, $c = 0; $c < $t; $c++) {
            $d += $cpf{$c} * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf{$c} != $d) {
            return false;
        }
    }
    return true;

}

function verificaEmailExistente($email,$tipo) {
    if($tipo == 'usuario')
        $email = User::where('email','=',$email)->first();
    else if($tipo=='cliente')
        $email = Client::where('mail','=',$email)->first();

    return ($email);
}

function verificaDocumentoExistente($doc) {
    $doc = Client::where('taxnumber','=',$doc)->first();
    return ($doc);
}

function usuariosCliente($cliente) {
    $users = User::select('id','name','email','active','client_id', 'role_id')
                ->where('client_id','=',$cliente)
                ->orderBy('active','desc')
                ->orderBy('name','asc')
                ->get();
    return $users;
}

function sendCurl($postfields, $method = null) {
    $sps = getSPS();

    $credentials = base64_encode($sps->key.':'.$sps->secret);

    $curl = curl_init();

    if( env('FITBANK_ENV') == 'dev' ) $FITBANK_URL = env('FITBANK_SANDBOX');
    else $FITBANK_URL = env('FITBANK_PROD');

    $postfieldsDecoded = json_decode($postfields, true);

    if (array_key_exists('Identifier', $postfieldsDecoded)) {
	    $postfieldsDecoded['Identifier'] = $postfieldsDecoded['Identifier'] . microtime(true);
	}
	$postfields = json_encode($postfieldsDecoded);


	curl_setopt_array($curl, array(
		CURLOPT_URL => $FITBANK_URL,
		CURLOPT_RETURNTRANSFER => true,
		CURLOPT_ENCODING => "",
		CURLOPT_MAXREDIRS => 10,
		CURLOPT_TIMEOUT => 5,
		CURLOPT_FOLLOWLOCATION => true,
		CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
		CURLOPT_CUSTOMREQUEST => "POST",
		CURLOPT_POSTFIELDS =>$postfields,
		CURLOPT_HTTPHEADER => array(
			"Content-Type: application/json",
			"Authorization: Basic ".$credentials
		),
	));

	// LOGS CURL FORAM REMOVIDOS POIS A ROTINA EM DEV (PGTO MENSALIDADE) NÃO É EXECUTADA POR UM USUÁRIO
	$log_id = Log::saveLogCurl($postfields, $method);

	//ZNE20220120 $response = json_decode( curl_exec($curl),true );
	$response = curl_exec($curl);

	$response_decoded = json_decode($response, true );

    // dd( curl_errno($curl) );
    curl_close($curl);

    // LOGS CURL FORAM REMOVIDOS POIS A ROTINA EM DEV (PGTO MENSALIDADE) NÃO É EXECUTADA POR UM USUÁRIO
    $log_id = Log::saveLogCurl($postfields, $method, $response, $log_id);

    //ZNE20220120 return $response;
    return $response_decoded;
}

function getLastCollectSmartsafe() {
    $client = Client::where('id','=',Auth::user()->client_id)->first();

    $transaction = Transactionlog::select('actiondate')
                            ->where('method','=','MoneyTransferIn')
                            ->where('client_id','=',$client->id)
                            ->where('success','=','1')
                            ->where('taxnumber','=',$client->taxnumber)
                            ->orderBy('actiondate','desc')
                            ->first();

    $dt = (empty($transaction->actiondate))?'20190101000000':somenteNumeros($transaction->actiondate);
    $dataref = substr($dt,0,4).'-'.substr($dt,4,2).'-'.substr($dt,6,2).'-'.substr($dt,8,2).'-'.substr($dt,10,2).'-'.substr($dt,12,2);

    $credentials = base64_encode($client->taxnumber);

    $curl = curl_init();

    if( env('FITBANK_ENV') == 'dev' ) $SMARTSAFE_URL = env('SMARTSAFE_SANDBOX');
    else $SMARTSAFE_URL = env('SMARTSAFE_PROD');

    curl_setopt_array($curl, array(
      CURLOPT_URL => $SMARTSAFE_URL."/sps/".$client->taxnumber."/".$dataref,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_ENCODING => "",
      CURLOPT_MAXREDIRS => 10,
      CURLOPT_TIMEOUT => 5,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
      CURLOPT_CUSTOMREQUEST => "GET",
      CURLOPT_HTTPHEADER => array(
        "Content-Type: application/json",
        "Authorization: ".$credentials
      ),
    ));

    $response = json_decode( curl_exec($curl),true );
    curl_close($curl);
    return $response;
}

function entryDate2dataHora($entrydate) {
    $dt = explode('T',$entrydate);
    return dataUs2Br($dt[0]).' - '.substr($dt[1], 0, 5);
}

function entryDate2data($entrydate) {
    $dt = explode('T',$entrydate);
    return date2br($dt[0]);
}

function date2us($databr)
{
    if(empty($databr)) return true;
    $dataus = explode('/',$databr);
    return $dataus[2].'-'.$dataus[1].'-'.$dataus[0];
}

function dataUs2Br($data) {
    return implode('/',(array_reverse(explode('-', $data))));
}

function date2br($dataus)
{
    if(empty($dataus)) return '';
    $databr = explode('-',$dataus);
    return $databr[2].'/'.$databr[1].'/'.$databr[0];
}

function datetime2br($datefull) {

    $dateshort = explode(' ', $datefull);

    return date2br($dateshort[0]).' - '.substr($dateshort[1],0,5);
}

function datetime2databr($datefull) {

    $dateshort = explode(' ', $datefull);

    return date2br($dateshort[0]);
}

function dataSlashReverse($data) {
    return implode('/',(array_reverse(explode('/', $data))));
}


function getSPS() {
    $info = Spsinfo::select('*')->first();

    $spsbank = Spsbank::where('spsinfo_id','=',$info->id)->first();

    $info->supplierphone = somenteNumeros($info->supplierphone);
    $info->banknumber = somenteNumeros($spsbank->bankcode);
    $info->bankbranch = somenteNumeros($spsbank->bankbranch);
    $info->bankaccount = somenteNumeros($spsbank->bankaccount);

    return $info;
}

function getOperationType($type) {
    switch ($type) {
            case '0':  $operation = 'Info'; break;
            case '1':  $operation = 'Fechamento De Conta'; break;
            case '2':  $operation = 'Tarifa Para Fechamento Conta'; break;
            case '3':  $operation = 'Tarifa Ativacao'; break;
            case '5':  $operation = 'Saque'; break;
            case '6':  $operation = 'Tarifa Saque Automatico'; break;
            case '7':  $operation = 'Tarifa Repasse Para Rede'; break;
            case '8':  $operation = 'Tarifa Consulta Saldo'; break;
            case '9':  $operation = 'Pagamento Conta'; break;
            case '10': $operation = 'Tarifa Pagamento Conta'; break;
            case '11': $operation = 'Recarga Dinheiro'; break;
            case '12': $operation = 'Recarga Celular Pre-Pago'; break;
            case '13': $operation = 'Perdas Operação'; break;
            case '14': $operation = 'PagamentoChargebackAoCliente'; break;
            case '15': $operation = 'Provisao Para Charge back'; break;
            case '16': $operation = 'Recarga Boleto'; break;
            case '17': $operation = 'Ajuste Conta Feito Pelo SAC'; break;
            case '18': $operation = 'Pagamento Chargeback'; break;
            case '19': $operation = 'Tarifa Cobrada Pelo Nao Uso Do Cartao'; break;
            case '20': $operation = 'Tarifa Uso Internacional'; break;
            case '21': $operation = 'Tarifa Atendimento Por Atendente CallCenter'; break;
            case '22': $operation = 'TarifaRecargaValoresCartão'; break;
            case '23': $operation = 'Promoçoes Marketing'; break;
            case '24': $operation = 'Tarifa Transferencia Entre Cartoes'; break;
            case '25': $operation = 'Transferencia Entre Cartoes'; break;
            case '26': $operation = 'Compra Via POS Com Cartao'; break;
            case '27': $operation = 'Tarifa Compras'; break;
            case '29': $operation = 'Taxa Uso Internacional'; break;
            case '30': $operation = 'Remessa Internacional Valores'; break;
            case '31': $operation = 'Tarifa Remessa Internacional'; break;
            case '32': $operation = 'Juros Conta Investimento'; break;
            case '33': $operation = 'Transferencia Para Conta Investimento'; break;
            case '34': $operation = 'Tarifa Emissão Extrato Enviado Pelo Correio'; break;
            case '35': $operation = 'Pacote Compras'; break;
            case '36': $operation = 'Imposto Sobre Vendas'; break;
            case '37': $operation = 'Compras Atraves Celular'; break;
            case '38': $operation = 'Tarifa Recarga Valores Cartao'; break;
            case '39': $operation = 'IOF'; break;
            case '40': $operation = 'Tarifa Cartao Secundario'; break;
            case '41': $operation = 'Saque Em Caixa Automatico Internacional'; break;
            case '42': $operation = 'Compras Internacionais Via POS Com Cartao'; break;
            case '43': $operation = 'Transacoes Desconhecidas'; break;
            case '44': $operation = 'Retencao Cargas Em Excesso'; break;
            case '45': $operation = 'Liberacao Cargas Em Excesso'; break;
            case '46': $operation = 'Recargas Nao Identificadas'; break;
            case '47': $operation = 'Recargas Retidas'; break;
            case '48': $operation = 'Tarifa Saque Lotericas'; break;
            case '49': $operation = 'Pagamento Contas'; break;
            case '50': $operation = 'TED'; break;
            case '51': $operation = 'Recarga Loterica'; break;
            case '52': $operation = 'Tarifa TED'; break;
            case '53': $operation = 'Saque Loterica'; break;
            case '54': $operation = 'Tarifa Recarga Lotericas'; break;
            case '55': $operation = 'Chargeback'; break;
            case '56': $operation = 'Recarga Online'; break;
            case '57': $operation = 'Debito Automatico'; break;
        default:
                $operation = '-';
                break;
    }
    return $operation;
}

function formatMoneyBr($money) {
    if(empty($money)) return '0,00';
    return number_format($money,2,',','.');
}

function covert2Float($value) {

    $value = clearMoney($value);

    return ($value=='')?0: str_replace(',', '.', str_replace('.', '', $value));
}

/**
 * Função adicionada para corrigir a função covert2Float()
 * Converter qualquer string com valor monetário e converte para float no padrão americano (com . sendo separador de decimais)
 * @param $value
 * @return float
 */
function convertToFloat($value): float
{
    $value = clearMoney($value);

    $valorFloat = str_replace(['R$', '.', ','], '', $value);
    return floatval(str_replace('.', '', substr($valorFloat, 0, -2)) . '.' . substr($valorFloat, -2));
}

function saveTransactionLog($client_id,$taxnumber,$method,$datasent,$success,$message,$validation,$entry,$balance,$actiondate,$cashbackvalue=0,$paymentdate='0000-00-00') {

    $user_id = (!Auth::user())?0:Auth::user()->id;

    $internalidentifier = ($message == 'Transferencia interna')?'transferenciainterna':'';

    $salvalog = new Transactionlog();
    $salvalog->user_id = $user_id;
    $salvalog->client_id = $client_id;
    $salvalog->taxnumber = $taxnumber;
    $salvalog->method = $method;
    $salvalog->datasent = $datasent;
    $salvalog->success = $success;
    $salvalog->message = $message;
    $salvalog->validation = $validation;
    $salvalog->entry = $entry;
    $salvalog->balance = $balance;
    $salvalog->actiondate = $actiondate;
    $salvalog->moneytransferindocumentnumber = $entry;
    $salvalog->documentnumber = $entry;
    $salvalog->ratevalue = $cashbackvalue;
    $salvalog->paymentdate = $paymentdate;
    $salvalog->internalidentifier = $internalidentifier;
    $salvalog->status = ($method=='InternalTransfer')?'Transferido':'0';

    $salvalog->save();

    return true;
}

function savePaymentLog($taxnumber, $duedate ,$paymentdate ,$method ,$identifier ,$transactionnumber, $barcode ,$datasent ,$success ,$message ,$validation ,$entryid ,$url ,$value ,$discount ,$extravalue) {
    $userInfo = Auth::user();

    $salvalog = new Paymentlog();
    $salvalog->user_id = $userInfo->id;
    $salvalog->client_id = $userInfo->client_id;
    $salvalog->taxnumber = $taxnumber;
    $salvalog->duedate = $duedate;
    $salvalog->paymentdate = $paymentdate;
    $salvalog->method = $method;
    $salvalog->identifier = $identifier;
    $salvalog->barcode = $barcode;
    $salvalog->datasent = $datasent;
    $salvalog->success = $success;
    $salvalog->message = $message;
    $salvalog->validation = $validation;
    $salvalog->entryid = $entryid;
    $salvalog->url = $url;
    $salvalog->value = $value;
    $salvalog->discount = $discount;
    $salvalog->extravalue = $extravalue;
    // $salvalog->status = '0';
    $salvalog->actiondate = date('Y-m-d H:i:s');

    $salvalog->save();

    return true;
}

function saveTransferManager($taxnumber, $documento, $nome, $banco, $agencia, $conta, $digito, $tipoconta, $valor, $ratevalue, $identificador, $success, $documentnumber, $url, $alreadyexists, $limiteespecial, $valorlimiteespecial, $datasent) {
    $userInfo = Auth::user();

    //$salvalog = new Transferlog();
    $salvalog = new Transfermanager();

    $salvalog->user_id = $userInfo->id;
    $salvalog->client_id = $userInfo->client_id;
    $salvalog->actiondate = date('Y-m-d H:i:s');

    $salvalog->taxnumber = $taxnumber;
    $salvalog->nome = $nome;
    $salvalog->documento = $documento;
    $salvalog->valor = $valor;
    $salvalog->identificador = $identificador;

    $salvalog->tipoconta = $tipoconta;
    $salvalog->banco = $banco;
    $salvalog->agencia = $agencia;
    $salvalog->conta = $conta;
    $salvalog->digito = $digito;
    $salvalog->limiteespecial = $limiteespecial;

    $salvalog->ratevalue = $ratevalue;
    $salvalog->identifier = $identificador;
    $salvalog->success = $success;
    $salvalog->documentnumber = $documentnumber;
    $salvalog->url = $url;
    $salvalog->alreadyexists = $alreadyexists;
    $salvalog->valorlimiteespecial = $valorlimiteespecial;
    $salvalog->datasent = $datasent;

    $salvalog->status = 'Incluido';
    $salvalog->save();

    $result['id'] = $salvalog->id;

    return $result;
}

function saveTransferLog(
    $taxnumber,
    $documento,
    $nome,
    $banco,
    $agencia,
    $conta,
    $digito,
    $tipoconta,
    $valor,
    $ratevalue,
    $identificador,
    $success,
    $documentnumber,
    $url,
    $alreadyexists,
    $limiteespecial,
    $valorlimiteespecial,
    $datasent,
    $paymentDate = null,
    $isScheduled = 0
) {
    // $userInfo = Auth::user();
    $userInfo = (object)[];

    if ( !Auth::user() ) {
        $userInfo->id = 1;
        $userInfo->client_id = 1;
    } else {
        $userInfo = Auth::user();
    }

    $salvalog = new Transferlog();
    $salvalog->user_id = $userInfo->id;
    $salvalog->client_id = $userInfo->client_id;
    $salvalog->taxnumber = $taxnumber;
    $salvalog->actiondate = date('Y-m-d H:i:s');
    $salvalog->documento = $documento;
    $salvalog->nome = $nome;
    $salvalog->banco = $banco;
    $salvalog->agencia = $agencia;
    $salvalog->conta = $conta;
    $salvalog->digito = $digito;
    $salvalog->tipoconta = $tipoconta;
    $salvalog->valor = $valor;
    $salvalog->ratevalue = $ratevalue;
    $salvalog->identificador = $identificador;
    $salvalog->identifier = $identificador;
    $salvalog->success = $success;
    $salvalog->documentnumber = $documentnumber;
    $salvalog->url = $url;
    $salvalog->alreadyexists = $alreadyexists;
    $salvalog->limiteespecial = $limiteespecial;
    $salvalog->valorlimiteespecial = $valorlimiteespecial;
    $salvalog->datasent = $datasent;
    $salvalog->paymentdate = $paymentDate;
    $salvalog->is_scheduled = $isScheduled;
    $salvalog->scheduled_status = $isScheduled === 1 ? 0 : null;
    $salvalog->status = '0';

    $salvalog->save();

    return true;
}


function saveTransferErrorLog($method, $datasent, $message, $validation,$idclient=false) {
    $user_id = (!Auth::user())?0:Auth::user()->id;

    if(!$idclient) {
        $client_id = (!Auth::user())?0:Auth::user()->client_id;
    } else {
        $client_id = $idclient;
    }

    $salvalog = new Transfererrorlog();
    $salvalog->user_id = $user_id;
    $salvalog->client_id = $client_id;
    $salvalog->method = $method;
    $salvalog->actiondate = date('Y-m-d H:i:s');
    $salvalog->datasent = $datasent;
    $salvalog->message = $message;
    $salvalog->validation = $validation;

    $salvalog->save();

    return true;
}

function saveBillet($clientcontact_id, $taxnumber, $customername, $customertaxnumber, $customermail,
        $customerphone, $addressline1, $addressline2, $neighborhood, $city, $state, $zipcode, $mailtosend,
        $phonetosend, $externalnumber, $identifier, $comments, $duedate, $totalvalue, $finedate, $finepercent,
        $finevalue, $interestpercent, $interestvalue, $discountdate, $discountvalue, $rebatevalue, $documentnumber,
        $entryid, $url, $alreadyexists, $datasent, $boleto,$ratevalue) {


    $userInfo = Auth::user();

    $billet = new Billet();
    $billet->user_id = $userInfo->id;
    $billet->client_id = $userInfo->client_id;
    $billet->clientcontact_id = $clientcontact_id;
    $billet->taxnumber = $taxnumber;
    $billet->emission = date('Y-m-d H:i:s');
    $billet->customername = $customername;
    $billet->customertaxnumber = $customertaxnumber;
    $billet->customermail = $customermail;
    $billet->customerphone = $customerphone;
    $billet->addressline1 = $addressline1;
    $billet->addressline2 = $addressline2;
    $billet->neighborhood = $neighborhood;
    $billet->city = $city;
    $billet->state = $state;
    $billet->zipcode = $zipcode;
    $billet->mailtosend = $mailtosend;
    $billet->phonetosend = $phonetosend;
    $billet->externalnumber = $externalnumber;
    $billet->identifier = $identifier;
    $billet->comments = $comments;
    $billet->duedate = $duedate;
    $billet->totalvalue = $totalvalue;
    $billet->finedate = $finedate;
    $billet->finepercent = $finepercent;
    $billet->finevalue = $finevalue;
    $billet->interestpercent = $interestpercent;
    $billet->interestvalue = $interestvalue;
    $billet->discountdate = $discountdate;
    $billet->discountvalue = $discountvalue;
    $billet->rebatevalue = $rebatevalue;
    $billet->documentnumber = $documentnumber;
    $billet->entryid = $entryid;
    $billet->url = $url;
    $billet->alreadyexists = $alreadyexists;
    $billet->datasent = $datasent;
    $billet->boleto = $boleto;
    $billet->status = '0';
    $billet->ratevalue = $ratevalue;
    $billet->save();

    return true;
}

function saveBilleterrorLog($taxnumber, $customername, $customertaxnumber, $totalvalue, $message, $datasent, $validation) {
    $userInfo = Auth::user();

    $logerror = new Billeterror();
    $logerror->user_id = $userInfo->id;
    $logerror->client_id = $userInfo->client_id;
    $logerror->taxnumber = $taxnumber;
    $logerror->customername = $customername;
    $logerror->customertaxnumber = $customertaxnumber;
    $logerror->totalvalue = $totalvalue;
    $logerror->emission = date('Y-m-d H:i:s');
    $logerror->message = $message;
    $logerror->datasent = $datasent;
    $logerror->validation = $validation;
    $logerror->save();

    return true;
}

function getClientById($id) {

    if($id==0) return false;

    $client = Client::where('id','=',$id)->first();

    return $client;
}

function getClientByTaxnumber($taxnumber) {

    $client = Client::where('taxnumber','=',$taxnumber)
        ->where('active', '=', '1')
        ->first();

    if(!$client) {
        $multi = Multisafe::where('taxnumber', '=', $taxnumber)
            ->where('active', '=', '1')
            ->first();
        $client = Client::where('id','=',$multi->client_id)
            ->where('active', '=', '1')
            ->first();
    }

    return $client;
}

function getUserById($id) {

    if($id==0) return false;

    $user = User::where('id','=',$id)->first();

    return $user;
}

function getFitbankBalance() {
    $spsInfo = getSPS();
    if(Auth::user()->client_id == 0) {
        $documento = $spsInfo->cnpj;
    } else {
        $client = getClientById( Auth::user()->client_id );
        $documento = $client->taxnumber;
    }


    $curlParams = "{ \"Method\": \"GetAccountEntryPaged\",
                        \"PartnerId\": ".$spsInfo->partnerid.",
                        \"BusinessUnitId\": ".$spsInfo->businessunitid.",
                        \"TaxNumber\": \"".$documento."\",
                        \"OnlyBalance\":\"true\"
                    }";

    $extrato = sendCurl($curlParams);
    $msg = false;
    if($extrato["Success"] == "false") return 0;

    return $extrato['Balance'];
}

function verificaStatusTransferencia($docnumber) {
    $spsInfo = getSPS();
    $curlParams = "{ \"Method\": \"GetMoneyTransferOutById\",
                    \"PartnerId\": ".$spsInfo->partnerid.",
                    \"BusinessUnitId\": ".$spsInfo->businessunitid.",
                    \"DocumentNumber\": \"".$docnumber."\"
                }";

    $check = sendCurl($curlParams);

    if($check["Success"] == "false") return false;

    return json_decode($check['Transferencia']);
}

function getTransferStatus($code) {
    switch ($code) {
        case '0':
                $status = 'Criado';
            break;

        case '1':
                $status = 'Enviado';
            break;

        case '2':
                $status = 'Pago';
            break;

        case '3':
                $status = 'Aguardando envio';
            break;

        case '4':
                $status = 'Sem saldo';
            break;

        case '5':
                $status = 'Cancelado';
            break;

        default:
                $status = 'Indefinido';
            break;
    }
    return $status;
}

function getBoletoStatus($code) {
    /*0 - Criado (Cobrança foi gerada no nosso sistema)
    1 - Pode ser registrado ('Pronto para envio ao banco')
    2 - Registrando ('enviado para o banco')
    3 - Registrado
    4 - Rejeitado (Foi rejeitado pelo banco por algum motivo)
    5 - Liquidado (Pago)
    7 - Cancelado (Cancelado pelo nosso sistema, banco ou pela chamada na API)
    8 - Erro Interno (Erro no nosso sistema)
    9 - Erro de Saldo*/
    switch ($code) {
        case '0':
                $status = 'Criado';
            break;

        case '1':
                $status = 'Pronto para envio ao banco';
            break;

        case '2':
                $status = 'Enviado para o banco';
            break;

        case '3':
                $status = 'Registrado';
            break;

        case '4':
                $status = 'Rejeitado';
            break;

        case '5':
                $status = 'Liquidado';
            break;

        case '6':
                $status = '-';
            break;

        case '7':
                $status = 'Cancelado';
            break;

        case '8':
                $status = 'Erro Interno';
            break;

        case '9':
                $status = 'Erro de Saldo';
            break;

        default:
                $status = 'Indefinido';
            break;
    }
    return $status;
}

function getDarjStatus($code) {
    /*0 - Criado (Cobrança foi gerada no nosso sistema)
    4 - Rejeitado (Foi rejeitado pelo banco por algum motivo)
    5 - Liquidado (Pago)
    7 - Cancelado (Cancelado pelo nosso sistema, banco ou pela chamada na API)
    8 - Erro Interno (Erro no nosso sistema)*/
    switch ($code) {
        case '0':
                $status = 'Criado';
            break;

        case '4':
                $status = 'Rejeitado';
            break;

        case '5':
                $status = 'Liquidado';
            break;

        case '7':
                $status = 'Cancelado';
            break;

        case '8':
                $status = 'Erro Interno';
            break;

        default:
                $status = 'Indefinido';
            break;
    }
    return $status;
}

function checkDocNumberTransfer($docnumber) {
    $check = Transferlog::where('documentnumber','=',$docnumber)->first();
    return (!$check)?false:true;
}

function checkDocNumberTransferIn($docnumber,$metodo = '') {
    $check = Transactionlog::where('entry','=',$docnumber)->where('method','=',$metodo)->first();
    return (!$check)?false:true;
}

/*  **************** DARF ********************/
function darfConsultar($docnumber) {
    $spsInfo = getSPS();
    $curlParams = "{ \"Method\": \"GetDarfOutById\",
                \"PartnerId\": ".$spsInfo->partnerid.",
                \"BusinessUnitId\": ".$spsInfo->businessunitid.",
                \"DocumentNumber\": \"".$docnumber."\"
            }";

    $enviar = sendCurl($curlParams);

    if($enviar["Success"] == "false") return false;


}

function saldoCliente($taxnumber) {
    $spsInfo = getSPS();

    $curlParams = "{ \"Method\": \"GetAccountEntryPaged\",
        \"PartnerId\": ".$spsInfo->partnerid.",
        \"BusinessUnitId\": ".$spsInfo->businessunitid.",
        \"TaxNumber\": \"".$taxnumber."\",
        \"OnlyBalance\":\"true\"
    }";

    $extrato = sendCurl($curlParams);

    $saldo = ($extrato["Success"] == "false") ? 0 : $extrato['Balance'];
    return $saldo;
}


function saldoClientePaged($taxnumber, $page = 1, $pagesize = 10) {
    $datade  = date('Y/m/d', strtotime('-7 days'));
    $dataate = date('Y/m/d');
    $spsInfo = getSPS();

    $curlParams = "{ \"Method\": \"GetAccountEntryPaged\",
        \"PartnerId\": ".$spsInfo->partnerid.",
        \"BusinessUnitId\": ".$spsInfo->businessunitid.",
        \"TaxNumber\": \"".$taxnumber."\",
        \"StartDate\": \"".$datade."\",
        \"EndDate\": \"".$dataate."\",
        \"PageSize\": ".$pagesize.",
        \"PageIndex\": ".$page.",
        \"OnlyBalance\":\"true\"
    }";

    $extrato = sendCurl($curlParams);

    $saldo = ($extrato["Success"] == "false") ? 0 : $extrato['Balance'];
    return $saldo;
}

// transfere da conta mãe para o cliente ( saldo disponível)
function transfereSaldoDisponivelCliente($taxnumber, $valor, $mensagem = '', $cashbackvalue = 0, $cli_ratevalue = 0, $userIsMaster = false) {
    $client = Client::where('taxnumber','=',$taxnumber)->first();
    $multi = false;

    if(!$client) {
        $multi = Multisafe::where('taxnumber', '=', somenteNumeros($taxnumber))->first();

        $client = Client::where('id', '=', $multi->client_id)->first();
    }

    $spsInfo = getSPS();
    $clientSpsService = new ClientSpsService(somenteNumeros($taxnumber));
    $saldoRealAntes = $clientSpsService->__get('saldoReal');

    $dataAtual = date("Y/m/d");
    $estampa = date("YmdHis");
    $clienteIdentifier = str_pad($client->id, 4, "0", STR_PAD_LEFT);

    $valorATransferir = ( $valor + $cli_ratevalue + $cashbackvalue);

    $curlParams = "{ \"Method\": \"InternalTransfer\",
                \"PartnerId\": ".$spsInfo->partnerid.",
                \"BusinessUnitId\": ".$spsInfo->businessunitid.",
                \"FromTaxNumber\": \"".somenteNumeros($spsInfo->cnpj)."\",
                \"ToTaxNumber\": \"".somenteNumeros($taxnumber)."\",
                \"Value\": \"".$valorATransferir."\",
                \"RateValue\": 0,
                \"TransferDate\": \"".$dataAtual."\",
                \"Identifier\": \"".$clienteIdentifier . $estampa."\",
                \"Description\": \"".$mensagem."\",
                \"CashBackValue\":\"".$cashbackvalue."\"
            }";
    // CashBackValue é o valor cobrado do beneficiário. Deve SEMPRE ser menor que o valor da transferência

    $transfere = sendCurl($curlParams);

    $transactionNumber = (isset($transfere['DocumentNumber']))?$transfere['DocumentNumber']:'0';
    $sucesso = ($transfere["Success"] == "false")?0:1;
    $validacao = (isset($transfere['Validation']))?serialize($transfere['Validation']):'';
    $mensagem = (isset($transfere['Message']))?$transfere['Message']:'';

    $checkSaved = checkDocNumberTransferIn($transactionNumber,'InternalTransfer');

    if(!$checkSaved) saveTransactionLog($client->id,somenteNumeros($taxnumber),'InternalTransfer',$curlParams,$sucesso,$mensagem,$validacao,$transactionNumber,$valorATransferir,date('Y-m-d H:i:s'),$cashbackvalue);

    if($transfere["Success"] == "false") {
        $validacao = (isset($transfere['Validation']))?serialize($transfere['Validation']):'';
        $mensagem = (isset($transfere['Message']))?$transfere['Message']:'';

        saveTransferErrorLog('InternalTransfer', $curlParams, $mensagem, $validacao);

        return false;
    }

    if(!$multi) {
        $client->availablelimit = (($client->availablelimit - $valorATransferir) < 0)?0:($client->availablelimit - $valorATransferir);
        $client->update();
    } else {
        $multi->availablelimit = (($multi->availablelimit - $valorATransferir) < 0)?0:($multi->availablelimit - $valorATransferir);
        $multi->update();
    }

    // aguarda FitBank creditar na conta do cliente | normal
    if($transfere["Success"] == "true" && $userIsMaster == false) {
        $attempts = 5;
        $delay = 5;

        for ($i = 0; $i < $attempts; $i++) {
            $saldoRealAtual = $clientSpsService->__get('saldoReal');

            if ($saldoRealAtual < $saldoRealAntes + $valor) {
                sleep($delay);
            } else {
                break;
            }
        }
    }

    return $transactionNumber;
}



// transfere da conta mãe para o cliente ( saldo disponível)
function transfereSaldoEntreContas($taxnumberOrigem, $taxnumberDestino, $valor, $mensagem = '', $cashbackvalue = 0, $cli_ratevalue = 0) {

    $clientOrigem = Client::where('taxnumber','=',$taxnumberOrigem)->first();
    $multiOrigem = false;
    if(!$clientOrigem) {
        $multiOrigem = Multisafe::where('taxnumber', '=', somenteNumeros($taxnumberOrigem))->first();
        $clientOrigem = Client::where('id', '=', $multiOrigem->client_id)->first();
    }

    $clientDestino = Client::where('taxnumber','=',$taxnumberDestino)->first();
    $multiDestino = false;
    if(!$clientOrigem) {
        $multiDestino = Multisafe::where('taxnumber', '=', somenteNumeros($taxnumberDestino))->first();
        $clientDestino = Client::where('id', '=', $multiDestino->client_id)->first();
    }

    $spsInfo = getSPS();

    $dataAtual = date("Y/m/d");
    $estampa = date("YmdHis");
    $random2dig = str_pad(random_int(0, 99), 2, "0", STR_PAD_LEFT);
    $clienteIdentifier = str_pad($clientOrigem->id, 4, "0", STR_PAD_LEFT);

    $valorATransferir = ( $valor + $cli_ratevalue + $cashbackvalue);

    $curlParams = "{ \"Method\": \"InternalTransfer\",
                \"PartnerId\": ".$spsInfo->partnerid.",
                \"BusinessUnitId\": ".$spsInfo->businessunitid.",
                \"FromTaxNumber\": \"".somenteNumeros($taxnumberOrigem)."\",
                \"ToTaxNumber\": \"".somenteNumeros($taxnumberDestino)."\",
                \"Value\": \"".$valorATransferir."\",
                \"RateValue\": 0,
                \"TransferDate\": \"".$dataAtual."\",
                \"Identifier\": \"".$clienteIdentifier . $estampa .  $random2dig . "\",
                \"Description\": \"".$mensagem."\",
                \"CashBackValue\":\"".$cashbackvalue."\"
            }";
    // CashBackValue é o valor cobrado do beneficiário. Deve SEMPRE ser menor que o valor da transferência

    $transfere = sendCurl($curlParams);

    $transactionNumber = (isset($transfere['DocumentNumber']))?$transfere['DocumentNumber']:'0';
    $sucesso = ($transfere["Success"] == "false")?0:1;
    $validacao = (isset($transfere['Validation']))?serialize($transfere['Validation']):'';
    $mensagem = (isset($transfere['Message']))?$transfere['Message']:'';

    $checkSaved = checkDocNumberTransferIn($transactionNumber,'EntreContas'.somenteNumeros($taxnumberDestino));

    if(!$checkSaved) saveTransactionLog($clientOrigem->id,somenteNumeros($taxnumberOrigem),'EntreContas'.somenteNumeros($taxnumberDestino),$curlParams,$sucesso,$mensagem,$validacao,$transactionNumber,$valorATransferir,date('Y-m-d H:i:s'),$cashbackvalue);

    if($transfere["Success"] == "false") {
        $validacao = (isset($transfere['Validation']))?serialize($transfere['Validation']):'';
        $mensagem = (isset($transfere['Message']))?$transfere['Message']:'';

        saveTransferErrorLog('EntreContas', $curlParams, $mensagem, $validacao);

        return false;
    }

    return $transfere;
}

function arrayObj2string($arr) {
    $resp = '';
    foreach($arr as $k => $v) {
        $resp .= $k .'::'. $v .'|';
    }
    return rtrim($resp,'|');
}

function zeroNull($number) {
    return (!empty($number))?$number:0;
}

function concatIdentifier($identifier) {
    $str = strtotime("now");
    return $str.'__'.$identifier;
}

function detectDateFormatOutputDateBr($datain,$retorno='date') {
    $pos = strpos($datain, '/');
    if($pos !== false) {
        $dt = explode('/', $datain);
        if(strlen($dt[0])>2) return implode('/', array_reverse($dt));
        return $datain;
    }

    $posT = strpos($datain, 'T');
    if($posT !== false) {
        if(($posT !== false) && $retorno == 'datetime' ) return entryDate2dataHora($datain);
        if(($posT !== false) && $retorno == 'date' ) return entryDate2data($datain);
    }

    $posD = strpos($datain, '-');
    $posC = strpos($datain, ':');
    if(($posD !== false) && ($posC !== false)) {
        if($retorno == 'datetime' ) return datetime2br($datain);
        if($retorno == 'date' ) return datetime2databr($datain);
    } else if(($posD !== false) && ($posC === false)) {
        return dataUs2Br($datain);
    }
    return '';
}

function traduzStatus($status) {
    switch ($status) {
        case 'Paid':
                $str = '<span class="text-success">Pago</span>';
            break;

        case 'Incluido':
                $str = '<span class="text-info">Incluído</span>';
            break;

        case 'Autorizado':
                $str = '<span class="text-success">Autorizado</span>';
            break;

        case 'Cancel':
                $str = '<span class="text-danger">Cancelado</span>';
            break;

        case 'Cancelled':
                $str = '<span class="text-danger">Cancelado</span>';
            break;

        case 'Registered':
                $str = '<span class="text-info">Registrado</span>';
            break;

        case 'Rejected':
                $str = '<span class="text-danger">Rejeitado</span>';
            break;

        case 'ErrorBalance':
                $str = '<span class="text-danger">Erro de saldo</span>';
            break;

        case 0:
                $str = '<span class="text-danger">Aberto</span>';
            break;

        case 1:
                $str = '<span class="text-success">Pago</span>';
            break;

        default:
                $str = '<span class="text-info">Aguardando</span>';
            break;
    }
    return $str;
}

function traduzStatusApi($status) {
    switch ($status) {
        case 'Paid':
                $str = 'Pago';
            break;

        case 'Cancel':
                $str = 'Cancelado';
            break;

        case 'Cancelled':
                $str = 'Cancelado';
            break;

        case 'Registered':
                $str = 'Registrado';
            break;

        case 'Rejected':
                $str = 'Rejeitado';
            break;

        case 'ErrorBalance':
                $str = 'Erro de saldo';
            break;

        default:
                $str = 'Aguardando';
            break;
    }
    return $str;
}



function traduzMetodoLogErro($metodo) {
    switch ($metodo) {
        case 'GenerateBoletoOut':
                $str = 'Pagto boletos';
            break;

        case 'GeneratePaymentDARJ':
                $str = 'Pagto DARJ';
            break;

        case 'GeneratePaymentDARF':
                $str = 'Pagto DARF';
            break;

        case 'GeneratePaymentFGTS':
                $str = 'Pagto FGTS';
            break;

        case 'GeneratePaymentGPS':
                $str = 'Pagto GPS';
            break;

        case 'GeneratePaymentGARE':
                $str = 'Pagto GARE';
            break;

        default:
                $str = $metodo;
            break;
    }
    return $str;
}

function traduzSucessoLog($valor) {
    switch ($valor) {
        case '0':
                $str = 'Falha';
            break;

        case '1':
                $str = 'Sucesso';
            break;

        default:
                $str = 'Nulo';
            break;
    }
    return $str;
}

/* **********************************************

    formata um numero de telefone

 ********************************************** */
function formataTelefone($num) {
    return '('.substr($num,0,2).') '.substr($num,2,4).'-'.substr($num,6);
}

/* **********************************************

    salva log de acesso do usuário

 ********************************************** */
function logAcesso($user_id, $accessorigin, $iporigin, $other) {
    $log = new Loginaccesslog();
    $log->user_id = $user_id;
    $log->accessdate = date('Y-m-d H:i:s');
    $log->accessorigin = $accessorigin;
    $log->iporigin = $iporigin;
    $log->other = $other;
    $log->save();
    return true;
}

/* **********************************************

    retorna o último acesso anterior do usuário

 ********************************************** */
function getUserLastAccess() {
    $last = Loginaccesslog::select('accessdate')->where('user_id','=',Auth::user()->id)->orderBy('accessdate','desc')->skip(1)->first();
    return (!$last)?'-':datetime2br($last->accessdate);
}

/* **********************************************

    salva os detalhes da utilização do adiantamento de saldo

 ********************************************** */
function logAvailableLimitUsed($user_id, $client_id, $taxnumber, $availablelimitInicial, $requestedvalue, $transactiontype, $documentnumber, $logid, $internaltransferdocumentnumber, $ratevalue, $duedate = false) {

    if(!$logid) {
        $log = new Usedavailablelimit();
        $log->user_id = $user_id;
        $log->client_id = $client_id;
        $log->emission = date('Y-m-d H:i:s');
        $log->taxnumber = $taxnumber;
        $log->availablelimit = $availablelimitInicial;
        $log->requestedvalue = $requestedvalue;
        $log->transactiontype = $transactiontype;
        $log->documentnumber = $documentnumber;
        $log->internaltransferdocumentnumber = $internaltransferdocumentnumber;
        $log->status = 'Paid';
        $log->ratevalue = $ratevalue;
        $log->duedate = ($duedate)
            ? $duedate
            : date('Y-m-d', strtotime("+1 day")) . ' 09:00:00' ;
        $log->save();

        return $log->id;
    } else {
        $log = Usedavailablelimit::findOrFail($logid);
        $log->documentnumber = $documentnumber;
        $log->update();

        return true;
    }
}

/* **********************************************

    retorna as transferencias do smartsafe > sps

 ********************************************** */
function getSmartsafeTransfers($client) {
    $transfers = Smartsafetransfer::where('client_id','=',$client)
                                    ->get();

    return $transfers;
}

/* **********************************************

    verifica se a transação utilizou adiantamento de saldo disponicel

 ********************************************** */
function verificaUtilizadoAdiantamento($docnumber, $tipo) {
    $tipoDarf  = strpos($tipo, 'Darf');
    $tipoDarj  = strpos($tipo, 'Darj');
    $tipoFgts  = strpos($tipo, 'Fgts');
    $tipoGps   = strpos($tipo, 'Gps');
    $tipoGare  = strpos($tipo, 'Gare');
    $tipoTrans = strpos($tipo, 'PreDischargeAccount');
    $tipoPagto = strpos($tipo, 'PreDebitBoletoOut');

    $res = false;

    if($tipoDarf !== false) $res = Darfpayment::select('id', 'valorlimiteespecial')->where('documentnumber','=',$docnumber)->where('limiteespecial','=',1);
    if($tipoDarj !== false) $res = Darjpayment::select('id', 'valorlimiteespecial')->where('documentnumber','=',$docnumber)->where('limiteespecial','=',1);
    if($tipoFgts !== false) $res = Fgtspayment::select('id', 'valorlimiteespecial')->where('documentnumber','=',$docnumber)->where('limiteespecial','=',1);
    if($tipoGps  !== false) $res = Gpspayment::select('id', 'valorlimiteespecial')->where('documentnumber','=',$docnumber)->where('limiteespecial','=',1);
    if($tipoGare !== false) $res = Garepayment::select('id', 'valorlimiteespecial')->where('documentnumber','=',$docnumber)->where('limiteespecial','=',1);
    if($tipoTrans!== false) $res = Transferlog::select('id', 'valorlimiteespecial')->where('documentnumber','=',$docnumber)->where('limiteespecial','=',1);
    if($tipoPagto!== false) $res = Barcodepayment::select('id', 'valorlimiteespecial')->where('documentnumber','=',$docnumber)->where('limiteespecial','=',1);

    if(!$res) return false;

    $resp = $res->first();
    return (!$resp)?false:$resp->valorlimiteespecial;
}

/* **********************************************

    verifica se o intenal transfer é referente a um adiantamento de saldo (admin)

 ********************************************** */
function verificaAdiantamento($docnumber) {
    $adiantamento = Usedavailablelimit::select('id', 'internaltransferdocumentnumber')
                                        ->where('internaltransferdocumentnumber','=',$docnumber)
                                        ->first();
    return (!$adiantamento)?false:$adiantamento->id;
}


/* **********************************************

    desfaz o procedimento de adiantamento em caso de cancelamento da transação
    - cancela a internalTransfer
    - soma de volta o saldo disponível que foi subtraído do cliente
    - atualiza o status do log de saldo utilizado para 'cancel'

 ********************************************** */
function desfazAdiantamento($docnumber, $tipo) {
    $tipoDarf  = strpos($tipo, 'Darf');
    $tipoDarj  = strpos($tipo, 'Darj');
    $tipoFgts  = strpos($tipo, 'Fgts');
    $tipoGps   = strpos($tipo, 'Gps');
    $tipoGare  = strpos($tipo, 'Gare');
    $tipoTrans = strpos($tipo, 'MoneyTransfer');
    $tipoPagto = strpos($tipo, 'BoletoOut');

    $type = false;

    if($tipoDarf !== false) $type = 'DARF';
    if($tipoDarj !== false) $type = 'DARJ';
    if($tipoFgts !== false) $type = 'FGTS';
    if($tipoGps  !== false) $type = 'GPS';
    if($tipoGare !== false) $type = 'GARE';
    if($tipoTrans!== false) $type = 'TRANSFERENCIA';
    if($tipoPagto!== false) $type = 'BOLETO';

    $adiantamento = Usedavailablelimit::select('id', 'client_id', 'internaltransferdocumentnumber','documentnumber','requestedvalue')
                                    ->where('documentnumber','=',$docnumber)
                                    ->where('transactiontype','=',$type)
                                    ->first();

    if($adiantamento) {
        $adiantamento->status = 'Cancel';
        $adiantamento->update();

        cancelaInternalTransfer($adiantamento->internaltransferdocumentnumber);

        $atualizaLimiteDisponivelCliente = Client::where('id','=',$adiantamento->client_id)->first();
        $atualizaLimiteDisponivelCliente->availablelimit = ($atualizaLimiteDisponivelCliente->availablelimit + $adiantamento->requestedvalue);
        $atualizaLimiteDisponivelCliente->update();

        return true;
    }
    // TODO: se falso logar erro

    return true;
}

/* **********************************************

    cancela a internalTransfer

 ********************************************** */
function cancelaInternalTransfer($documentnumber) {
    $spsInfo = getSPS();
    $curlParams = "{ \"Method\": \"CancelInternalTransfer\",
               \"PartnerId\": ".$spsInfo->partnerid.",
                \"BusinessUnitId\": ".$spsInfo->businessunitid.",
               \"DocumentNumber\": \"".$documentnumber."\"
           }";

   $enviar = sendCurl($curlParams);

   if($enviar["Success"] == "false") {
       // TODO: logar erro
   }

   return true;
}


/* **********************************************

    consulta a taxa a ser cobrada do cliente, se será a padrão ou a diferenciada do cliente
    Transaction: internaltransfer, moneytransfer, boletoout, payment
 ********************************************** */
function consultaTaxaCliente($client_id, $transaction, $transactionvalue) {
    $client = Client::where('id','=',$client_id)->first();

    $taxrate = 0;

    if($client->customrates == '1') {

        switch ($transaction) {
            case 'internaltransfer':
                    if($client->balanceusagepercentage == '1') $taxrate = (($client->balanceusage * $transactionvalue)/100);
                    else $taxrate = $client->balanceusage;
                break;

            case 'moneytransfer':
                    if($client->transferspercentage == '1') $taxrate = (($client->transfers * $transactionvalue)/100);
                    else $taxrate = $client->transfers;
                break;

            case 'boletoout':
                    $taxrate = $client->billets;
                break;

            case 'payment':
                    if($client->paymentspercentage == '1') $taxrate = (($client->payments * $transactionvalue)/100);
                    else $taxrate = $client->payments;
                break;

            default:
                    $taxrate = 0;
                break;
        }

    } else {
        $sps = getSPS();

        switch ($transaction) {
            case 'internaltransfer':
                    if($sps->clientbalanceusagepercentage == '1') $taxrate = (($sps->clientbalanceusage * $transactionvalue)/100);
                    else $taxrate = $sps->clientbalanceusage;
                break;

            case 'moneytransfer':
                    if($sps->clienttransferspercentage == '1') $taxrate = (($sps->clienttransfers * $transactionvalue)/100);
                    else $taxrate = $sps->clienttransfers;
                break;

            case 'boletoout':
                    $taxrate = $sps->clientbillets;
                break;

            case 'payment':
                    if($sps->clientpaymentspercentage == '1') $taxrate = (($sps->clientpayments * $transactionvalue)/100);
                    else $taxrate = $sps->clientpayments;
                break;

            default:
                    $taxrate = 0;
                break;
        }
    }

    return $taxrate;
}

/* **********************************************

    lista bancos

 ********************************************** */
function getBanks() {
    $banks = Bank::select('id','name','code')->orderBy('name','ASC')->get();

    return $banks;
}

function getSpsBanks($id) {
    $banks = Spsbank::where('spsinfo_id','=',$id)->orderBy('nickname','ASC')->get();

    return $banks;
}

/**
 * Função que busca em usedavailablelimits se existe uma mensalidade lançada para o cliente
 * Retorna independente se foi paga ou não, filtrando pelo mês atual
 *
 * @param string $client_id Id do cliente na tabela clients
 * @param string $mesAtual Mês atual
 *
 * @return boolean
 */
function verificarSeExisteMensalidade($taxnumber, $mesAtual) {
    $existeMensalidade = Usedavailablelimit::where([
        'taxnumber' => $taxnumber,
        'transactiontype' => 'MENSALIDADE'
        ])
        ->whereRaw("MONTH(emission) = ?", $mesAtual)
        ->first();

    return (!$existeMensalidade) ? false : true;
}

function getClients() {
    $clients = Client::select('id','name','taxnumber')->orderBy('name','ASC')->get();

    return $clients;
}

function getMes($num) {
    switch ($num) {
        case '01':
                $mes = 'Janeiro';
            break;

        case '02':
                $mes = 'Fevereiro';
            break;

        case '03':
                $mes = 'Março';
            break;

        case '04':
                $mes = 'Abril';
            break;

        case '05':
                $mes = 'Maio';
            break;

        case '06':
                $mes = 'Junho';
            break;

        case '07':
                $mes = 'Julho';
            break;

        case '08':
                $mes = 'Agosto';
            break;

        case '09':
                $mes = 'Setembro';
            break;

        case '10':
                $mes = 'Outubro';
            break;

        case '11':
                $mes = 'Novembro';
            break;

        case '12':
                $mes = 'Dezembro';
            break;

        default:
                $mes = '-';
            break;
    }
    return $mes;
}

// function getTaxFee($feetype, $client=0) {

//     $sps = getSPS();

//     switch ($feetype) {
//         case 'pagamento':
//             # code...
//             break;

//         case 'transferencia':
//             # code...
//             break;

//         case 'boleto':
//             # code...
//             break;

//         case 'mensalidade':
//             # code...
//             break;

//         default:
//             # code...
//             break;
//     }

// }

function calcFee($percent, $value, $fee) {
    $val = $fee;
    if($percent == 1) {
        $val = ($value*$fee)/100;
    }
    return $val;
}

function clearMoney($val) {
    return preg_replace( '/[^0-9\.,]/is', '', $val );
}

function operationTimeLimit($operation) {
    $spsInfo = Spsinfo::select('id','paymentstimelimit','transferstimelimit','billetstimelimit')->first();

    $overlimit = true;

    date_default_timezone_set("America/Sao_Paulo");

    switch ($operation) {
        case 'pagamento':
                $overlimit = (time() < strtotime( $spsInfo->paymentstimelimit.':00' ));
            break;

        case 'transferencia':
                $overlimit = (time() < strtotime( $spsInfo->transferstimelimit.':00' ));
            break;

        case 'boleto':
                $overlimit = (time() < strtotime( $spsInfo->billetstimelimit.':00' ));
            break;

        default:
                $overlimit = true;
            break;
    }

    return $overlimit;

}

function isWeekend($date) {
    $dt = date('D',strtotime(date2us($date)));
    return ($dt=='Sat' || $dt=='Sun')?true:false;
}

/* ADICIONA DIA UTIL */
function addWeekDays($date, $amount=1) {
    return date('Y-m-d', strtotime($date. ' + '.$amount.' weekdays'));
}

/*  */
function isToday($date) {
    $fornecida=date("Y-m-d",strtotime(date2us($date)));
    $today = date("Y-m-d");
    return ($fornecida<=$today)?true:false;
}

function covertFloat2Br($value) {

    return ($value=='')?0: str_replace('.', ',', $value);
}

function getLatestFinancesSmartsafe($client_id) {
    $client = Client::where('id','=',$client_id)->first();

    $transaction = Transactionlog::select('actiondate')
                            ->where('method','=','InternalTransfer')
                            ->where('client_id','=',$client->id)
                            ->where('success','=','1')
                            ->where('taxnumber','=',$client->taxnumber)
                            ->where('internalidentifier','=','transferenciainterna')
                            ->orderBy('actiondate','desc')
                            ->first();

    if(!$transaction) return [];

    $dataref = somenteNumeros($transaction->actiondate);;

    $credentials = base64_encode($client->taxnumber);

    $curl = curl_init();

    if( env('FITBANK_ENV') == 'dev' ) $SMARTSAFE_URL = env('SMARTSAFE_SANDBOX');
    else $SMARTSAFE_URL = env('SMARTSAFE_PROD');

    curl_setopt_array($curl, array(
      CURLOPT_URL => $SMARTSAFE_URL."/sps/movimentacoes/".$client->taxnumber."/".$dataref,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_ENCODING => "",
      CURLOPT_MAXREDIRS => 10,
      CURLOPT_TIMEOUT => 5,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
      CURLOPT_CUSTOMREQUEST => "GET",
      CURLOPT_HTTPHEADER => array(
        "Content-Type: application/json",
        "Authorization: ".$credentials
      ),
    ));

    customLog('curl call: Authorization: '.$credentials);
    customLog('curl call: '.$SMARTSAFE_URL."/sps/movimentacoes/".$client->taxnumber."/".$dataref);

    if(curl_errno($curl)){
        customLog('Deu ruim curl: '.curl_error($curl));
        return [];
    }

    $response = json_decode( curl_exec($curl),true );

    curl_close($curl);
    return ( isset($response) && array_key_exists('success', $response) && !$response['success'])?[]:$response;
}


function daysBetweenDates($presentdate, $pastdate) {

    $present = date('Y-m-d', strtotime($presentdate));
    $past    = date('Y-m-d', strtotime($pastdate));

    return round( (strtotime($present) - strtotime($past))  / (60 * 60 * 24));
}


function getClientBalance($taxnumber) {
    $spsInfo = getSPS();

    $curlParams = "{ \"Method\": \"GetAccountEntryPaged\",
                        \"PartnerId\": ".$spsInfo->partnerid.",
                        \"BusinessUnitId\": ".$spsInfo->businessunitid.",
                        \"TaxNumber\": \"".$taxnumber."\",
                        \"OnlyBalance\":\"true\"
                    }";

    $extrato = sendCurl($curlParams);
    $msg = false;
    if($extrato["Success"] == "false") return 0;

    return $extrato['Balance'];
}

// transfere da conta mãe para o cliente ( saldo disponível)
function transfereClienteParaSps($taxnumber, $valor, $mensagem = '', $cashbackvalue = 0) {
    //$client = Client::where('taxnumber','=',$taxnumber)->first();
    // trata  caso de multisafe
    $client = getClientByTaxnumber($taxnumber);

    $spsInfo = getSPS();

    $random2dig = str_pad(random_int(0, 99), 2, "0", STR_PAD_LEFT);
    $dataAtual = date("Y/m/d");
    $estampa = date("YmdHis");
    $clienteIdentifier = str_pad($client->id, 4, "0", STR_PAD_LEFT);

    $curlParams = "{ \"Method\": \"InternalTransfer\",
                \"PartnerId\": ".$spsInfo->partnerid.",
                \"BusinessUnitId\": ".$spsInfo->businessunitid.",
                \"FromTaxNumber\": \"".somenteNumeros($taxnumber)."\",
                \"ToTaxNumber\": \"".somenteNumeros($spsInfo->cnpj)."\",
                \"Value\": \"".$valor."\",
                \"RateValue\": 0,
                \"TransferDate\": \"".$dataAtual."\",
                \"Identifier\": \"".$clienteIdentifier . $estampa. "_juros adiantamento de saldo\",
                \"Description\": \"".$mensagem."\",
                \"CashBackValue\":\"".$cashbackvalue."\"
            }";
    // CashBackValue é o valor cobrado do beneficiário. Deve SEMPRE ser menor que o valor da transferência

    $transfere = sendCurl($curlParams);

    $transactionNumber = (isset($transfere['DocumentNumber']))?$transfere['DocumentNumber']:'0';
    $sucesso = ($transfere["Success"] == "false")?0:1;
    $validacao = (isset($transfere['Validation']))?serialize($transfere['Validation']):'';
    $mensagem = (isset($transfere['Message']))?$transfere['Message']:'';

    $checkSaved = checkDocNumberTransferIn($transactionNumber,'InternalTransfer');

    if(!$checkSaved) saveTransactionLog($client->id,somenteNumeros($client->taxnumber),'InternalTransfer',$curlParams,$sucesso,$mensagem,$validacao,$transactionNumber,$valor,date('Y-m-d H:i:s'),$cashbackvalue);

    if($transfere["Success"] == "false") {
        $validacao = (isset($transfere['Validation']))?serialize($transfere['Validation']):'';
        $mensagem = (isset($transfere['Message']))?$transfere['Message']:'';

        saveTransferErrorLog('InternalTransfer', $curlParams, $mensagem, $validacao);

        return false;
    }

    return $transactionNumber;
}

function getSmartsafeBalance($client_id) {
    $client = Client::where('id','=',$client_id)->first();

    $transaction = Transactionlog::select('actiondate')
                            ->where('method','=','InternalTransfer')
                            ->where('client_id','=',$client->id)
                            ->where('success','=','1')
                            ->where('taxnumber','=',$client->taxnumber)
                            ->where('internalidentifier','=','transferenciainterna')
                            ->orderBy('actiondate','desc')
                            ->first();

    if(!$transaction) return 0;

    $dataref = somenteNumeros($transaction->actiondate);;

    $credentials = base64_encode($client->taxnumber);

    $curl = curl_init();

    if( env('FITBANK_ENV') == 'dev' ) $SMARTSAFE_URL = env('SMARTSAFE_SANDBOX');
    else $SMARTSAFE_URL = env('SMARTSAFE_PROD');

    curl_setopt_array($curl, array(
      CURLOPT_URL => $SMARTSAFE_URL."/sps/saldo/".$client->taxnumber."/".$dataref,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_ENCODING => "",
      CURLOPT_MAXREDIRS => 10,
      CURLOPT_TIMEOUT => 5,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
      CURLOPT_CUSTOMREQUEST => "GET",
      CURLOPT_HTTPHEADER => array(
        "Content-Type: application/json",
        "Authorization: ".$credentials
      ),
    ));

    if(curl_errno($curl)){
        return false;
    }

    $response = json_decode( curl_exec($curl),true );

    curl_close($curl);

    if(isset($response) && $response['success']) {
        $client->availablelimit = $response['saldo'];
        $client->update();
    } else {
        $client->availablelimit = 0;
        $client->update();
    }

    return ( isset($response) && $response !=false  )?$response['saldo']:false;
}


function customLog($conteudo) {
    $filename = base_path().'/public/customlogs/log.txt';
    $carimbo = date('d-m-Y_His')."__";

    // Primeiro vamos ter certeza de que o arquivo existe e pode ser alterado
    if (is_writable($filename)) {

     // Em nosso exemplo, nós vamos abrir o arquivo $filename
     // em modo de adição. O ponteiro do arquivo estará no final
     // do arquivo, e é pra lá que $conteudo irá quando o
     // escrevermos com fwrite().
        if (!$handle = fopen($filename, 'a')) {
             // echo "Não foi possível abrir o arquivo ($filename)";
             exit;
        }

        // Escreve $conteudo no nosso arquivo aberto.
        if (fwrite($handle, $carimbo.$conteudo."\n") === FALSE) {
            // echo "Não foi possível escrever no arquivo ($filename)";
            exit;
        }

        // echo "Sucesso: Escrito ($conteudo) no arquivo ($filename)";

        fclose($handle);

    } else {
        // echo "O arquivo $filename não pode ser alterado";
    }
}

/* **********************************************

   atualiza o limite diponivel

 ********************************************** */
function updateAvailableLimit($client_id, $amount) {
    $client = Client::select('id','availablelimit')->where('id','=',$client_id)->first();
    $client->availablelimit = $amount;
    $client->update();
    return true;
}


function multiCofres($session_cnpj, $current_uri) {

    $client = Client::where('id','=',Auth::user()->client_id)->first();
    $multi = getMultiCnpj(Auth::user()->client_id);

    $clientTemp = Client::where('taxnumber','=',somenteNumeros($session_cnpj))->first();
    if(!$clientTemp) {
        $multiTemp = Multisafe::where('taxnumber', '=', somenteNumeros($session_cnpj))->first();
        $nome = $multiTemp->nickname;
    } else {
        $nome = $clientTemp->name;
    }

    $line = '<div class="row justify-content-md-center">
        <div class="col-md-8 col-12">
                    <div class="row  align-items-center">
                        <div class="col-md-6">Em uso: <strong>'. $nome .'</strong> ('.formataCNPJ($session_cnpj) . ')</div>
                        <div class="col-md-3">
                            <form class="form form-horizontal" action="/changecnpj" method="POST">
                                <input type="hidden" name="currenturi" value="'. $current_uri .'">
                                <input type="hidden" name="_token" value="'.csrf_token().'">
                                <select class="form-control" style="padding-top: 2px; padding-bottom: 2px; height: 26px;" name="alteraconta">
                                    <option value="'. $client->taxnumber .'" '. (($client->taxnumber == $session_cnpj)?'selected':'') .'>'. formataCNPJ($client->taxnumber) .'</option>';

                                    foreach($multi as $cnpj) {
                                        $line .= '<option value="'. $cnpj->taxnumber .'" '. (($cnpj->taxnumber == $session_cnpj)?'selected':'') .'>'. formataCNPJ($cnpj->taxnumber) .'</option>';
                                    }
    $line .= '                   </select>
                        </div>
                            <div class="col-md-3"><button type="submit" class="btn btn-light mr-1 mb-1 waves-effect waves-light" style="padding-top: 6px; padding-bottom: 6px; margin-top:8px;" >Trocar conta</button></div>
                    </div>
                </form>
        </div>
    </div><br>';

    return $line;

}


function getMultiCnpj($client_id = false) {
    $client = (!$client_id)? Auth::user()->client_id : $client_id;
    $multi = Multisafe::where('client_id', '=', $client)->get();
    return $multi;
}

function getAvailableLimitMultiCnpj($cnpj) {
    $multi = Multisafe::where('taxnumber', '=', somenteNumeros($cnpj))->first();

    if(!$multi) {
        $client = getClientByTaxnumber($cnpj);
        return (!$client)? 0 : $client->availablelimit;
    }

    return $multi->availablelimit;
}


function getAdiantamentoDevido($taxnumber = false) {
    //$clientId = (!$client_id)? Auth::user()->client_id : $client_id;

    // se taxnumber não for compatível com o tipo boolean OU for diferente de false
    if ($taxnumber !== false) {
        $limitesUtilizados = Usedavailablelimit::where('paid','=',0)
                                //->where('client_id','=',$clientId)
                                ->where('taxnumber','=',$taxnumber)
                                ->get();

        $saldoDevedor = 0;
        foreach($limitesUtilizados as $lim) {

            $saldoDevedor += $lim->requestedvalue
                             + $lim->ratevalue
                             - $lim->paymentvalue;

        }
        return $saldoDevedor;
    }

    // para erros na passagem do valor do taxnumber, retorna saldo devedor negativo
    return -1;
}

function getTaxaDevida($taxnumber = false) {
    //$clientId = (!$client_id)? Auth::user()->client_id : $client_id;

    // se taxnumber não for compatível com o tipo boolean OU for diferente de false
    if ($taxnumber !== false) {
        $limitesUtilizados = Usedavailablelimit::where('paid','=',0)
                                //->where('client_id','=',$clientId)
                                ->where('taxnumber','=',$taxnumber)
                                ->get();

        $saldoDevedor = 0;
        foreach($limitesUtilizados as $lim) {

            $saldoDevedor += $lim->interestvalue;

        }
        return $saldoDevedor;
    }

    // para erros na passagem do valor do taxnumber, retorna saldo devedor negativo
    return -1;
}

/*
    1. o parametro da função deve ser o CNPJ em vez do client_id
    2. para toda linha com paid = 0:
        FÓRMULA : saldoDevedor = requestedvalue + ratevalue - paymentvalue (+ interestvalue???)
        ATT : o juros entrará como uma nova linha => alterar a função de cálculo de juros
*/
function getSaldoDevedor($taxnumber = false) {
    //$clientId = (!$client_id)? Auth::user()->client_id : $client_id;

    // se taxnumber não for compatível com o tipo boolean OU for diferente de false
    if ($taxnumber !== false) {
        $limitesUtilizados = Usedavailablelimit::where('paid','=',0)
                                //->where('client_id','=',$clientId)
                                ->where('taxnumber','=',$taxnumber)
                                ->get();

        $saldoDevedor = 0;
        foreach($limitesUtilizados as $lim) {

            $saldoDevedor += $lim->requestedvalue
                             + $lim->ratevalue
                             - $lim->paymentvalue
                             + $lim->interestvalue;

        }
        return $saldoDevedor;
    }

    // para erros na passagem do valor do taxnumber, retorna saldo devedor negativo
    return -1;
}

function saveRealCollect($taxnumber, $realcollectedvalue, $transferedvalue, $transactionnumber) {

    $userInfo = Auth::user();

    $salvaDados = new CollectsMap();
    $salvaDados->user_id = $userInfo->id;
    $salvaDados->client_id = $userInfo->client_id;
    $salvaDados->taxnumber = $taxnumber;
    $salvaDados->emission = date('Y-m-d H:i:s');
    $salvaDados->realcollectedvalue = $realcollectedvalue;
    $salvaDados->transferedvalue = $transferedvalue;
    $salvaDados->transactionnumber = $transactionnumber;
    //$salvaDados->paymentvalue = $paymentvalue;
    //$salvaDados->differencevalue = $differencevalue;

    $salvaDados->save();

    return $salvaDados->id;
}

// Função antiga da Roseane
function pagarSaldoDevedor_Zane($collectID, $taxnumber, $disponivelParaPagamento) {

    $totalPago = 0;
    // se taxnumber não for compatível com o tipo boolean OU for diferente de false
    if ($taxnumber !== false) {
        $limitesUtilizados = Usedavailablelimit::where('paid','=',0)
                                ->where('taxnumber','=',$taxnumber)
                                ->orderBy('id','asc')
                                ->get();

        // Para cada debito, usa dinheiro disponível p/ quitaçao ou pagto parcial
        foreach($limitesUtilizados as $lim) {
            // - calcula o valor devido
            $valorDevido = $lim->requestedvalue
                           + $lim->ratevalue
                           - $lim->paymentvalue
                           + $lim->interestvalue;
            // - calcula a quitaçao ou o pagamento parcial
            if ($disponivelParaPagamento >= $valorDevido) {
                $valorPago = $valorDevido;
                $paid = 1;
            } else {
                $valorPago = $disponivelParaPagamento;
                $paid = 0;
            }
            // - atualiza paymentvalue, dinheiro disponível e o total pago
            $lim->paid = $paid;
            $lim->paymentvalue += $valorPago;
            if ($paid) $lim->paymentdate = date('Y-m-d H:i:s');
            $lim->update();
            $disponivelParaPagamento -= $valorPago;
            $totalPago += $valorPago;
            // - atualiza histórico de utilização da coleta real
            $pagtoID = savePaidUsedLimits($collectID, $lim->id, $valorPago);
            // - finaliza processo se acabou o dinheiro
            if ($disponivelParaPagamento <= 0) return $totalPago;
        }
    }

    return $totalPago;
}

// Função do Wallace, baseada na função da Roseane, inserido em 12 Jul 22
/**
 * Paga mensalidade
 * @param string $taxNumber Numero que identifica o cliente
 * @param int $disponivelParaPagamento Valor disponível para pagamento da dívida
 * @return int $totalPago Valor pago do saldo devedor
 */
function pagarSaldoDevedor(/*$collectID = null,*/ $taxnumber, $disponivelParaPagamento) {

    $totalPago = 0;

    // se taxnumber não for compatível com o tipo boolean OU for diferente de false
    if ($taxnumber !== false) {

        // Para o sprint de 18 Jul 22, essa query foi restrita para apenas mensalidade
        // Futuramente, pagará tudo
        $limitesUtilizados = Usedavailablelimit::where('paid', 0)
            ->where('taxnumber', $taxnumber)
            ->where('transactiontype', 'MENSALIDADE')
            ->orderBy('id', 'asc')
            ->get();

        // Para cada debito, usa dinheiro disponível p/ quitaçao ou pagto parcial
        foreach($limitesUtilizados as $lim) {

            // - calcula o valor devido
            $valorDevido = $lim->requestedvalue
                + $lim->ratevalue
                - $lim->paymentvalue
                + $lim->interestvalue;

            // - calcula a quitaçao ou o pagamento parcial
            if ($disponivelParaPagamento >= $valorDevido) {
                $valorPago = $valorDevido;
                $paid = 1;
            } else {
                $valorPago = $disponivelParaPagamento;
                $paid = 0;
            }

            $pagou = transfereClienteParaSps($taxnumber, $valorPago, 'internaltransfer', 0);
            if (!$pagou) return $totalPago;

            // - atualiza paymentvalue, dinheiro disponível e o total pago
            $lim->paid = $paid;
            $lim->paymentvalue += $valorPago;
            if ($paid) $lim->paymentdate = date('Y-m-d H:i:s');
            $lim->update();

            $disponivelParaPagamento -= $valorPago;
            $totalPago += $valorPago;

            // - atualiza histórico de utilização da coleta real
            //if ($collectID) $pagtoID = savePaidUsedLimits($collectID, $lim->id, $valorPago);
            // - finaliza processo se acabou o dinheiro

            if ($disponivelParaPagamento <= 0) return $totalPago;
        }
    }

    return $totalPago;
}

/**
 * Transfere valor do cliente para SPS e atualiza o registro da usedavailablelimits.
 * @param $objLimite
 * @param $valorATransferirParaSPS
 * @return array Usedavailablelimit $objLimite, bool $pagou
 */
function finalizarPagamentoLimitesUtilizados($objLimite, $valorATransferirParaSPS, $valorProvisionado, $mensagem = 'internaltransfer')
{
    $transferiu = transfereClienteParaSps(
        $objLimite->taxnumber,
        $valorATransferirParaSPS,
        $mensagem,
        0
    );

    if (!$transferiu) {
        if ($valorProvisionado > 0) {
            $objLimite->paymentvalue += $valorProvisionado;
            $objLimite->paymentdate = date('Y-m-d H:i:s');

            // TODO: Lógica para caso o update não ocorra
            $objLimite->update();
        }

        return [
            'objLimite' => $objLimite,
            'pagou' => false
        ];
    }

    $objLimite->paymentvalue += $valorProvisionado + $valorATransferirParaSPS;
    $objLimite->paymentdate = date('Y-m-d H:i:s');

    // TODO: Definir nome dessa variavel
    $valorPrincipal = $objLimite->requestedvalue + $objLimite->ratevalue + $objLimite->interestvalue;

    if ($valorPrincipal <= $objLimite->paymentvalue + 0.009) $objLimite->paid = 1;

    // TODO: Lógica para caso o update não ocorra
    $objLimite->update();

    return [
        'objLimite' => $objLimite,
        'pagou' => true
    ];
}
/**
 * @param object $objLimite Objeto do limite que está em pagamento
 * @param boolean $aplicarTaxaAdiantamento
 * @param boolean $aplicarTaxaNoProvisionamento
 * @return array 'valorATransferirParaSPS', 'valorProvisionado', 'idLimiteCriado'
 */
function prepararSaldoParaPagamentoLimitesUtilizados($objLimite, $aplicarTaxaAdiantamento = false, $aplicarTaxaNoProvisionamento = false)
{
    $valorProvisionado = 0;
    $valorATransferirParaSPS = 0;
    $idLimiteCriado = 0;

    $amanha = date('Y-m-d', strtotime("+1 day"));
    $amanhaNoveHoras = $amanha . " 09:00:00";

    $hoje = date('Y-m-d');
    $hojeNoveHoras = $hoje . " 09:00:00";

    $clientSpsService = new ClientSpsService($objLimite->taxnumber);

    // Existe ratevalue para manter o código compatível com futuras atualizações
    // em que o ratevalue poderá ser inserido na mensalidade
    $valorAPagar = $objLimite->requestedvalue + $objLimite->ratevalue + $objLimite->interestvalue - $objLimite->paymentvalue;

    $saldoCliente = saldoCliente($objLimite->taxnumber);

    $saldoCompensar = getAvailableLimitMultiCnpj($objLimite->taxnumber);
    $availablelimitInicial = $saldoCompensar;

    if ($saldoCliente >= $valorAPagar) {

        $valorATransferirParaSPS = $valorAPagar;

    } else {

        $valorAdiantamentoNecessario = $valorAPagar - $saldoCliente;

        if ($aplicarTaxaAdiantamento) {

            $taxaAdiantamento = consultaTaxaCliente(
                $objLimite->client_id,
                'internaltransfer',
                $saldoCompensar
            );

            $maiorAdiantamentoPossivel = $saldoCompensar - $taxaAdiantamento;

            $dataVencimento = $amanhaNoveHoras;

        } else {

            $maiorAdiantamentoPossivel = $saldoCompensar;
            $dataVencimento = $hojeNoveHoras;

        }

        if ($maiorAdiantamentoPossivel >= $valorAdiantamentoNecessario) {

            // A taxa de adiantamento é zerada porque o cliente pode receber dinheiro durante o dia
            // neste caso, não teria taxa de permanência
            // Então sempre será adiantado o valor necessário ou o máximo de saldo a compensar
            // que o cliente possuir
            $valorASerAdiantado = $valorAdiantamentoNecessario;

            $precisaProvisionar = false;

        } else {

            $valorASerAdiantado = $maiorAdiantamentoPossivel;

            $precisaProvisionar = true;

            $valorProvisionado = $valorAdiantamentoNecessario - $valorASerAdiantado;

        }

        $valorATransferirParaSPS = $valorASerAdiantado + $saldoCliente;

        // Preciso adiantar?
        if ($valorASerAdiantado > 0) {

            $valorCashback = 0;
            if ($aplicarTaxaAdiantamento) {
                $valorCashback = consultaTaxaCliente(
                    $objLimite->client_id,
                    'internaltransfer',
                    $valorASerAdiantado
                );
            }

            $adiantouSaldoACompensar = transfereSaldoDisponivelCliente(
                $objLimite->taxnumber,
                $valorASerAdiantado,
                'Adiantamento para pagamento de limite',
                $valorCashback,
                0
            );

            if ($adiantouSaldoACompensar) {

                $internalTransferDocumentNumber = $adiantouSaldoACompensar;

                $idLimiteCriado = logAvailableLimitUsed(
                    1,
                    $objLimite->client_id,
                    $objLimite->taxnumber,
                    $availablelimitInicial,
                    $valorASerAdiantado,
                    'ADIANTAMENTO',
                    'SSG_' . $objLimite->id,
                    false,
                    $internalTransferDocumentNumber,
                    $valorCashback,
                    $dataVencimento
                );

            } else {

                $valorATransferirParaSPS -= $valorASerAdiantado;

            }

        }

        if ($precisaProvisionar) {

            $taxaDoProvisionamento = 0;
            $dataVencimentoProvisionamento = $hojeNoveHoras;

            if ($aplicarTaxaNoProvisionamento) {
                $taxaDoProvisionamento = $clientSpsService->calculaTaxaDaOperacao('ADIANTAMENTO', $valorProvisionado);
                $dataVencimentoProvisionamento = $amanhaNoveHoras;
            }

            $idProvisionamento = realizaProvisionamento($objLimite->id, $valorProvisionado, $taxaDoProvisionamento, $dataVencimentoProvisionamento);

            if (!$idProvisionamento) {
                $valorProvisionado = 0;
            }

        }

    }

    return [
        'valorATransferirParaSPS' => $valorATransferirParaSPS,
        'valorProvisionado' => $valorProvisionado,
        'idLimiteCriado' => $idLimiteCriado
    ];
}

function savePaidUsedLimits($collectID, $usedavailablelimitsID, $paymentvalue) {

    //$userInfo = Auth::user();

    $salvaDados = new PaidUsedLimits();
    $salvaDados->collectsmap_id = $collectID;
    $salvaDados->usedavailablelimits_id = $usedavailablelimitsID;
    $salvaDados->paymentvalue = $paymentvalue;

    $salvaDados->save();

    return $salvaDados->id;

}

function realizaProvisionamento_Zane($usedlimit_id, $interestvalue, $cashbackvalue, $hoje) {

    // não possui taxa a provisionar, finaliza
    if (0 == $interestvalue) {
        return 0;
    }

    $utilizado = Usedavailablelimit::where('id','=',$usedlimit_id)->first();

    $user_id = $utilizado->user_id;
    $client_id = $utilizado->client_id;
    $taxnumber = $utilizado->taxnumber;
    $availablelimit = 0;
    $requestedvalue = $interestvalue;
    $transactiontype = 'PROVISIONAMENTO';
    $documentnumber = 'SSG_' . $usedlimit_id;
    $logid = false;
    $internaltransferdocumentnumber = 0;
    // calcular o cashback
    //$cashbackvalue = consultaTaxaCliente($client_id, 'internaltransfer', $requestedvalue);
    $ratevalue = $cashbackvalue;

    $provisionamento = logAvailableLimitUsed($user_id, $client_id, $taxnumber, $availablelimit, $requestedvalue, $transactiontype, $documentnumber, $logid, $internaltransferdocumentnumber, $ratevalue);

    if ($provisionamento) {
        // atualizar os dados do principal, data de pagamento, taxa e dias utilizados
        $utilizado->paymentdate = $hoje;
        $utilizado->interestvalue = 0;
        $utilizado->daysused = 0;
        $utilizado->update();

        // ataulizar o documentnumber do provisionamento acrescentando o id
        $provisionado = Usedavailablelimit::where('id','=',$provisionamento)->first();
        $provisionado->documentnumber .= '_' . $provisionado->id;
        $provisionado->update();
    }

    return $provisionamento;
}

/**
 * Nova função de realizar provisionamento
 * @author Wallace Andrade
 */
function realizaProvisionamento($usedlimit_id, $interestvalue, $cashbackvalue, $dataVencimento = FALSE ) {

    // não possui taxa a provisionar, finaliza
    if (0 == $interestvalue) {
        return 0;
    }

    $utilizado = Usedavailablelimit::where('id','=',$usedlimit_id)->first();

    $user_id = $utilizado->user_id;
    $client_id = $utilizado->client_id;
    $taxnumber = $utilizado->taxnumber;
    $availablelimit = 0;
    $requestedvalue = $interestvalue;
    $transactiontype = 'PROVISIONAMENTO';
    $documentnumber = 'SSG_' . $usedlimit_id;
    $logid = false;
    $internaltransferdocumentnumber = 0;
    // calcular o cashback
    //$cashbackvalue = consultaTaxaCliente($client_id, 'internaltransfer', $requestedvalue);
    $ratevalue = $cashbackvalue;

    // A var $hoje foi inserida para ser adicionada ao campo duedate do novo usedavailablelimit
    $provisionamento = logAvailableLimitUsed(
        $user_id,
        $client_id,
        $taxnumber,
        $availablelimit,
        $requestedvalue,
        $transactiontype,
        $documentnumber,
        $logid,
        $internaltransferdocumentnumber,
        $ratevalue,
        (($dataVencimento) ? $dataVencimento : date('Y-m-d') . ' 23:59:59')
    );

    if ($provisionamento) {
        // atualizar os dados do principal, data de pagamento, taxa e dias utilizados
        $utilizado->paymentdate = now();
        $utilizado->interestvalue = 0;
        $utilizado->daysused = 0;
        $utilizado->update();

        // ataulizar o documentnumber do provisionamento acrescentando o id
        $provisionado = Usedavailablelimit::where('id','=',$provisionamento)->first();
        $provisionado->update();
    }

    return $provisionamento;
}

/**
 * Calcula o valor total provisionado do cliente
 *
 * @author Fabio
 */
function calculaValorTotalProvisionado($taxnumber) {

    $totalProvisionado = Usedavailablelimit::where('taxnumber','=',$taxnumber)
                                            ->where('transactiontype','=','PROVISIONAMENTO')
                                            ->where('paid', '=', 0)
                                            ->sum('requestedvalue');

    return $totalProvisionado;
}

/**
 * Recebe um valor para incrementar ou decrementar ao saldo a compensar
 * @param string $taxnumber taxnumber do cliente
 * @param float $valor valor a ser incrementado ou decrementado
 * @return array availablelimit, diferencaAoForcarZero, forcouZeroNoSaldoCompensar
 */
// function addOuSubValorNoSaldoCompensarPeloTaxnumber(string $taxnumber, float $valor) {

//     $objCliente = Client::where('taxnumber', '=', $taxnumber)
//         ->where('active', '=', 1)
//         ->first();

//     if (!$objCliente) {
//         $objCliente = Multisafe::where('taxnumber', '=', $taxnumber)
//             ->where('active', '=', 1)
//             ->first();

//         // TODO: caso não encontrou o cliente
//     }

//     $saldoCompensarOriginal = $objCliente->availablelimit;

//     $objCliente->availablelimit += $valor;

//     $forcouZero = false;
//     $diferenca = 0;
//     if ($objCliente->availablelimit <= 0) {
//         $forcouZero = true;
//         $objCliente->availablelimit = 0;

//         $diferenca = $valor - $saldoCompensarOriginal;
//     }

//     $objCliente->update();

//     // TODO: Necessário definir como retornará a informação de que o decremento forçou o saldo a ser zerado
//     return [
//         'availablelimit' => $objCliente->availablelimit,
//         'diferencaAoForcarZero' => $diferenca,
//         'forcouZeroNoSaldoCompensar' => $forcouZero
//     ];
// }

function buscarClienteOuMultisafePeloTaxnumber($taxnumber)
{
    $objCliente = Client::where('taxnumber', '=', $taxnumber)
    ->where('active', '=', 1)
    ->first();

    if (!$objCliente) {
        $objCliente = Multisafe::where('taxnumber', '=', $taxnumber)
            ->where('active', '=', 1)
            ->first();

        // TODO: caso não encontrou o cliente
    }

    return $objCliente;
}

/**
 * Efetua transferência de taxnumber cliente para destinatário externo
 * @param $fromTaxnumber string|int Taxnumber origem
 * @param $toTaxnumber string|int Taxnumber destino
 * @param $toNickname string|int Nickname destino
 * @param $toBank string|int Id do banco de destino
 * @param $toBankBranch string|int Agencia do banco de destino
 * @param $toBankAccount string|int Conta bancaria de destino
 * @param $toBankAccountDigit string|int Digito da conta bancaria de destino
 * @param $toAccountType string|int Tipo da conta de destino, sendo 0 = corrente e 1 = poupança
 * @param $valor float Valor da transferencia
 * @param $taxaCliente float Taxa paga pelo cliente
 * @param $identifier string|int Identificador da transação
 * @param $usouSaldoCompensar bool TRUE ou FALSE
 * @param $valorSaldoCompensarUsado float Valor do Saldo a Compensar utilizado
 * @param $data string Data da transação, formato YYYY/mm/dd
 * @return array ["status" => TRUE|FALSE, "mensagem", "documentNumber" => apenas caso status seja TRUE]
 */
function transferirParaDestinatarioExterno(
    $fromTaxnumber,
    $toTaxnumber,
    $toNickname,
    $toBank,
    $toBankBranch,
    $toBankAccount,
    $toBankAccountDigit,
    $toAccountType,
    $valor,
    $taxaCliente,
    $identifier,
    $usouSaldoCompensar,
    $valorSaldoCompensarUsado
)
{
    $spsInfo = getSPS();

    // TODO
    // Em produção não será "Ymdhis"
    // será "Ymd"
    $identificador = date('Ymdhis', strtotime("now")).'__'.$identifier;

    //busca no banco se há uma transferencia com o identificador informado
    $transferinfo = Transferlog::
        where('user_id','=', ( (!Auth::user()) ? 0 : Auth::user()->id))
        ->where('taxnumber','=',$fromTaxnumber)
        ->where('identificador','=',$identificador)
        ->first();

    //Se houver uma transferencia com o identificador, redireciona o usuário informando a situação
    if($transferinfo) {
        return [
            "Success" => FALSE,
            "ErrorCode" => "ERRO_TRANSFERENCIA-DUPLICADA",
            "Message" => "Já existe uma transferência com esse identificador."
        ];
    }

    $parametrosCurl = [
        "Method" => "MoneyTransfer",
        "PartnerId" => $spsInfo->partnerid,
        "BusinessUnitId" => $spsInfo->businessunitid,
        "FromTaxNumber" => somenteNumeros($fromTaxnumber),
        "ToTaxNumber" => somenteNumeros($toTaxnumber),
        "ToName" => $toNickname,
        "Bank" => somenteNumeros($toBank),
        "BankBranch" => somenteNumeros($toBankBranch),
        "BankAccount" => somenteNumeros($toBankAccount),
        "BankAccountDigit" => somenteNumeros($toBankAccountDigit),
        "AccountType" => somenteNumeros($toAccountType),
        "Value" => $valor,
        "RateValue" => (string)$taxaCliente,
        "Identifier" => $identificador,
        "RateValueType" => "0",
        "PaymentDate" => date('Y/m/d'),
        "Tags" => []
    ];

    $curlParams = json_encode($parametrosCurl);

    $retornoTransferencia = sendCurl($curlParams);

    $validacao = (isset($retornoTransferencia['Validation']))
        ? serialize($retornoTransferencia['Validation'])
        : '';

    $mensagem = $retornoTransferencia["Message"] ?? "";

    if ($retornoTransferencia["Success"] == "false") {

        saveTransferErrorLog('MoneyTransfer', $curlParams, $mensagem, $validacao);

        return [
            "Success" => FALSE,
            "ErrorCode" => "ERRO_TRANSFERENCIA-FALHOU",
            "Message" => "Falha na transferência."
        ];
    }

//    $transactionNumber = (isset($transfere['DocumentNumber']))?$transfere['DocumentNumber']:'0';
    $transactionNumber = $retornoTransferencia["DocumentNumber"] ?? "0";

    $sucesso = ($retornoTransferencia["Success"] == "false" || $retornoTransferencia["Success"] == "False" )
        ? 0 : 1;

//    $entryid = (isset($retornoTransferencia["EntryId"]))?$retornoTransferencia["EntryId"]:'';
    $entryid = $retornoTransferencia["EntryId"] ?? "";

//    $url = (isset($retornoTransferencia['Url']))?$retornoTransferencia['Url']:'';
    $url = $retornoTransferencia["Url"] ?? "";

    $alreadyExists = ($retornoTransferencia['AlreadyExists'] == "false" || $retornoTransferencia['AlreadyExists'] == "False")
        ? 0 : 1;

    $checkSaved = checkDocNumberTransfer($transactionNumber);

    if(!$checkSaved) {
        saveTransferLog(
            somenteNumeros($fromTaxnumber),
            somenteNumeros($toTaxnumber),
            $toNickname,
            somenteNumeros($toBank),
            somenteNumeros($toBankBranch),
            somenteNumeros($toBankAccount),
            somenteNumeros($toBankAccountDigit),
            somenteNumeros($toAccountType),
            $valor,
            $taxaCliente,
            $identificador,
            $sucesso,
            $transactionNumber,
            $url,
            $alreadyExists,
            $usouSaldoCompensar,
            $valorSaldoCompensarUsado,
            $curlParams
        );

        return [
            "Success" => TRUE,
            "Message" => "Transferência executada com sucesso.",
            "DocumentNumber" => $transactionNumber
        ];
    } else {
        return [
            "Success" => FALSE,
            "ErrorCode" => "ERRO_TRANSFERENCIA-DUPLICADA-FITBANK",
            "Message" => "Falha na transferência, número FitBank repetido."
        ];
    }
}


/**
 * Efetua transferência de taxnumber cliente para destinatário externo
 * @param $fromTaxnumber string|int Taxnumber origem
 * @param $toTaxnumber string|int Taxnumber destino
 * @param $toNickname string|int Nickname destino
 * @param $toBank string|int Id do banco de destino
 * @param $toBankBranch string|int Agencia do banco de destino
 * @param $toBankAccount string|int Conta bancaria de destino
 * @param $toBankAccountDigit string|int Digito da conta bancaria de destino
 * @param $toAccountType string|int Tipo da conta de destino, sendo 0 = corrente e 1 = poupança
 * @param $valor float Valor da transferencia
 * @param $taxaCliente float Taxa paga pelo cliente
 * @param $identifier string|int Identificador da transação
 * @param $usouSaldoCompensar bool TRUE ou FALSE
 * @param $valorSaldoCompensarUsado float Valor do Saldo a Compensar utilizado
 * @param $data string Data da transação, formato YYYY/mm/dd
 * @return array ["status" => TRUE|FALSE, "mensagem", "documentNumber" => apenas caso status seja TRUE]
 */
function transferirParaDestinatarioExternoComData(
    $fromTaxnumber,
    $toTaxnumber,
    $toNickname,
    $toBank,
    $toBankBranch,
    $toBankAccount,
    $toBankAccountDigit,
    $toAccountType,
    $valor,
    $taxaCliente,
    $identifier,
    $usouSaldoCompensar,
    $valorSaldoCompensarUsado,
    $data
)
{
    $spsInfo = getSPS();

    // TODO
    // Em produção não será "Ymdhis"
    // será "Ymd"
    $identificador = date('Ymdhis', strtotime("now")).'__'.$identifier;

    //busca no banco se há uma transferencia com o identificador informado
    $transferinfo = Transferlog::
        where('user_id','=', ( (!Auth::user()) ? 0 : Auth::user()->id))
        ->where('taxnumber','=',$fromTaxnumber)
        ->where('identificador','=',$identificador)
        ->first();

    //Se houver uma transferencia com o identificador, redireciona o usuário informando a situação
    if($transferinfo) {
        return [
            "Success" => FALSE,
            "ErrorCode" => "ERRO_TRANSFERENCIA-DUPLICADA",
            "Message" => "Já existe uma transferência com esse identificador."
        ];
    }

    $parametrosCurl = [
        "Method" => "MoneyTransfer",
        "PartnerId" => $spsInfo->partnerid,
        "BusinessUnitId" => $spsInfo->businessunitid,
        "FromTaxNumber" => somenteNumeros($fromTaxnumber),
        "ToTaxNumber" => somenteNumeros($toTaxnumber),
        "ToName" => $toNickname,
        "Bank" => somenteNumeros($toBank),
        "BankBranch" => somenteNumeros($toBankBranch),
        "BankAccount" => somenteNumeros($toBankAccount),
        "BankAccountDigit" => somenteNumeros($toBankAccountDigit),
        "AccountType" => somenteNumeros($toAccountType),
        "Value" => $valor,
        "RateValue" => (string)$taxaCliente,
        "Identifier" => $identificador,
        "RateValueType" => "0",
        "PaymentDate" => $data,
        "Tags" => []
    ];

    $curlParams = json_encode($parametrosCurl);

    $retornoTransferencia = sendCurl($curlParams);

    $validacao = (isset($retornoTransferencia['Validation']))
        ? serialize($retornoTransferencia['Validation'])
        : '';

    $mensagem = $retornoTransferencia["Message"] ?? "";

    if ($retornoTransferencia["Success"] == "false") {

        saveTransferErrorLog('MoneyTransfer', $curlParams, $mensagem, $validacao);

        return [
            "Success" => FALSE,
            "ErrorCode" => "ERRO_TRANSFERENCIA-FALHOU",
            "Message" => "Falha na transferência."
        ];
    }

//    $transactionNumber = (isset($transfere['DocumentNumber']))?$transfere['DocumentNumber']:'0';
    $transactionNumber = $retornoTransferencia["DocumentNumber"] ?? "0";

    $sucesso = ($retornoTransferencia["Success"] == "false" || $retornoTransferencia["Success"] == "False" )
        ? 0 : 1;

//    $entryid = (isset($retornoTransferencia["EntryId"]))?$retornoTransferencia["EntryId"]:'';
    $entryid = $retornoTransferencia["EntryId"] ?? "";

//    $url = (isset($retornoTransferencia['Url']))?$retornoTransferencia['Url']:'';
    $url = $retornoTransferencia["Url"] ?? "";

    $alreadyExists = ($retornoTransferencia['AlreadyExists'] == "false" || $retornoTransferencia['AlreadyExists'] == "False")
        ? 0 : 1;

    $checkSaved = checkDocNumberTransfer($transactionNumber);

    if(!$checkSaved) {
        saveTransferLog(
            somenteNumeros($fromTaxnumber),
            somenteNumeros($toTaxnumber),
            $toNickname,
            somenteNumeros($toBank),
            somenteNumeros($toBankBranch),
            somenteNumeros($toBankAccount),
            somenteNumeros($toBankAccountDigit),
            somenteNumeros($toAccountType),
            $valor,
            $taxaCliente,
            $identificador,
            $sucesso,
            $transactionNumber,
            $url,
            $alreadyExists,
            $usouSaldoCompensar,
            $valorSaldoCompensarUsado,
            $curlParams
        );

        return [
            "Success" => TRUE,
            "Message" => "Transferência executada com sucesso.",
            "DocumentNumber" => $transactionNumber
        ];
    } else {
        return [
            "Success" => FALSE,
            "ErrorCode" => "ERRO_TRANSFERENCIA-DUPLICADA-FITBANK",
            "Message" => "Falha na transferência, número FitBank repetido."
        ];
    }
}

function taxnumberEstaDuplicado($taxnumber)
{
    $qntClientes = Client::select('taxnumber')
        ->where('active', '1')
        ->where('taxnumber', '=', $taxnumber)
        ->count();

    $qntMultisafes = Multisafe::select('taxnumber')
        ->where('active', '1')
        ->where('taxnumber', '=', $taxnumber)
        ->count();

    if ( ($qntClientes + $qntMultisafes) >= 2 ) {
        return TRUE;
    }

    return FALSE;
}

/**
 * Retorna o próximo dia útil
 * @param string $data Formato Y-m-d
 * @return string $proximoDiaUtil Formato Y-m-d
 */
function obterProximoDiaUtil($data)
{
    $dt = DateTime::createFromFormat("Y-m-d", $data);
    $ehDataValida = ($dt !== false && !array_sum($dt::getLastErrors()));

    if (!$ehDataValida) {
        return FALSE;
    }

    if ( isWeekend(dataUs2Br(date("Y-m-d", strtotime( $data . "+1 day")))) ) {

        if ( isWeekend(dataUs2Br(date("Y-m-d", strtotime( $data . "+2 days")))) ) {

            if ( isWeekend(dataUs2Br(date("Y-m-d", strtotime( $data . "+3 days")))) ) {

                $proximoDiaUtil = date("Y-m-d", strtotime( $data . "+3 days"));

            } else {
                $proximoDiaUtil = date("Y-m-d", strtotime( $data . "+2 days"));
            }

        } else {
            $proximoDiaUtil = date("Y-m-d", strtotime( $data . "+2 days"));
        }

    } else {
        $proximoDiaUtil = date("Y-m-d", strtotime( $data . "+1 day"));
    }

    return $proximoDiaUtil;
}

function ehDiaUtil($data)
{
    return !isWeekend(dataUs2Br($data));
}
