<?php

namespace App\Http\Controllers;

use App\User;
use App\Client;
use App\Address;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;

class UserController extends Controller
{
    public function authenticate(Request $request)
    {
        $credentials = $request->only('email', 'password');

        try {
            if (! $token = JWTAuth::attempt($credentials)) {
                return response()->json(['success'=>false,'error' => 'invalid_credentials'], 400);
            }
        } catch (JWTException $e) {
            return response()->json(['success'=>false,'error' => 'could_not_create_token'], 500);
        }

        $userInfo = JWTAuth::user();

        $client = getClientById($userInfo->client_id);

        $response = array(
        	"user" => array(
        		"id"   => $userInfo->id,
		        "name" => $userInfo->name,
		        "email"=> $userInfo->email,
		        "role" => $userInfo->role_id,
		        "token"=> $token,
                "taxnumber" => (!$client)?'':$client->taxnumber
        	)
        );

        $partnames = explode(' ', $userInfo->name,2);

        $other = 'User-Agent: '.$request->header('User-Agent');
        $logAcesso = logAcesso($userInfo->id,'api',$request->getClientIp(),$other);

        return response()->json(
            [
                "success" => true,
                "token" => $token,
                "user" => [
                    "id" =>    $userInfo->id,
                    "name" =>  $userInfo->name,
                    "firstname" =>  $partnames[0],
                    "lastname" =>  (!isset($partnames[1]))?'':$partnames[1],
                    "email" => $userInfo->email,
                    "role" =>  $userInfo->role_id,
                    "taxnumber" => (!$client)?'':$client->taxnumber
                ]
            ]
        );
    }

    public function register(Request $request)
    {
            $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:6|confirmed',
        ]);

        if($validator->fails()){
                return response()->json($validator->errors()->toJson(), 400);
        }

        $user = User::create([
            'name' => $request->get('name'),
            'email' => $request->get('email'),
            'password' => Hash::make($request->get('password')),
        ]);

        $token = JWTAuth::fromUser($user);

        return response()->json(compact('user','token'),201);
    }

    public function getAuthenticatedUser()
    {
            try {

                if (! $user = JWTAuth::parseToken()->authenticate()) {
                        return response()->json(['user_not_found'], 404);
                }

            } catch (Tymon\JWTAuth\Exceptions\TokenExpiredException $e) {

                    return response()->json(['token_expired'], $e->getStatusCode());

            } catch (Tymon\JWTAuth\Exceptions\TokenInvalidException $e) {

                    return response()->json(['token_invalid'], $e->getStatusCode());

            } catch (Tymon\JWTAuth\Exceptions\JWTException $e) {

                    return response()->json(['token_absent'], $e->getStatusCode());

            }

            $userInfo = JWTAuth::user();

	        // $response = array(
	        // 	"user" => array(
	        // 		"id"   => $userInfo->id,
			      //   "name" => $userInfo->name,
			      //   "email"=> $userInfo->email,
			      //   "role" => $userInfo->role_id
	        // 	)
	        // );
	        $response = array(
	        	"user" => array(
	        		"id"   => $userInfo->id,
			        "name" => $userInfo->name,
			        "email"=> $userInfo->email,
			        "role" => $userInfo->role_id
	        	)
	        );
	        

            return response()->json(compact('response'));
    }


    /* PROFILE */
    public function profile() {
    	$userInfo = JWTAuth::user();

    	$profile = User::select(
    							'users.name', 'users.email','users.role_id',
								'addresses.addresstype', 'addresses.addressline', 'addresses.complement', 'addresses.neighborhood', 
											'addresses.cityname', 'addresses.state', 'addresses.zipcode', 'addresses.addressline2', 'addresses.country',
								'clients.taxnumber', 'clients.businessunitid', 'clients.mail', 'clients.phone', 'clients.nickname', 'clients.name as clientname'
							)
    						->leftJoin('clients','clients.id','=','users.client_id')
    						->leftJoin('addresses','addresses.businessunitid','=','clients.businessunitid')
    						->where('users.id','=',$userInfo->id)
    						->first();


    	array_walk_recursive($profile, "replaceNullValueWithEmptyString");

    	$profile->taxnumber = (strlen($profile->taxnumber)<12)?:formataCNPJ($profile->taxnumber);

    	return response()->json(compact('profile'));

    }

    public function logout() {
        JWTAuth::invalidate(true);

        return response()->json(
            [
                "success" => true,
                "message" => "Logout"
            ]
        );
    }
}