<?php

namespace App\Console\Commands;

use App\Autotransferreceiver;
use App\Autotransferschedule;
use App\Classes\ClientSPS\ClientSpsService;
use App\Client;
use App\Multisafe;
use Illuminate\Console\Command;
use App\Logerrorautotransferschedule;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class AgendarDxAutomatico extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'command:agendardxautomatico';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Agenda todos os D+x automáticos';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->executarAgendamento();
    }

    public function executarAgendamento()
    {
        $this->limparLog();
        $dataCompletaHoje = date('Y-m-d');
        $multisafesQuery = Multisafe::select('taxnumber', 'nickname AS name', 'd0time', 'd1time', 'd0automatic', 'd1automatic')
            ->where('active', 1)
            ->where(function ($query) {
                $query->where('d0automatic', 1)
                    ->orWhere('d1automatic', 1);
            });
        $clientsQuery = Client::select('taxnumber', 'name', 'd0time', 'd1time', 'd0automatic', 'd1automatic')
            ->where('active', 1)
            ->where(function ($query) {
                $query->where('d0automatic', 1)
                    ->orWhere('d1automatic', 1);
            });
        $todosClientesAtivos = $multisafesQuery->unionAll($clientsQuery)->get();
        foreach($todosClientesAtivos as $cliente) {
            $cliente->d0automatic == 1 ? $this->agendarD0($cliente, $dataCompletaHoje) : $this->agendarD1($cliente, $dataCompletaHoje);
        }
    }

    public function limparLog()
    {
        $logPath = storage_path('logs/agendardxautomatico.log');
        if(File::exists($logPath))
            File::put($logPath, '');
    }

    /**
     * Busca na tabela autotransferschedule se já foi agendado
     * @param object $objCliente Cliente, podendo ser do tipo Client ou Multisafe
     * @param string $dataAgendamento Data completa do agendamento
     * @return bool
     */
    public function verificarSeExisteAgendamento($objCliente, $dataAgendamento, $operacao)
    {
        $dataAgendamento = Carbon::parse($dataAgendamento)->format('Y-m-d');
        $agendamento = Autotransferschedule::where("fromtaxnumber", "=", $objCliente->taxnumber)
            ->where("operationtype", "=", $operacao)
            ->whereNotIn("operationstatus", ["CANCELADO"])
            ->where(DB::raw("DATE_FORMAT(scheduleddatetime, '%Y-%m-%d')"), $dataAgendamento)->get();
        return ($agendamento->count() > 0) ? TRUE : FALSE;
    }

    public function efetuarAgendamento($objCliente, $dataAgendamento, $operacao)
    {
        if ($this->verificarSeExisteAgendamento($objCliente, $dataAgendamento, $operacao)) {
            return FALSE;
        }

        try {
            $destinatario = Autotransferreceiver::where("fromtaxnumber", "=", $objCliente->taxnumber)
                ->where("operationtype", "=", $operacao)
                ->first();

            if (!$destinatario) {
                throw new \Exception('Destinatário não existente.');
            }

            $horarioDx = ($operacao == "D+0") ? $objCliente->d0time : $objCliente->d1time;


            $novoAgendamento = new Autotransferschedule();
            $novoAgendamento->fromtaxnumber = $destinatario->fromtaxnumber;

            $novoAgendamento->operationtype = $operacao;
            $novoAgendamento->scheduleddatetime = "${dataAgendamento} ${horarioDx}";

            $novoAgendamento->totaxnumber = $destinatario->totaxnumber;
            $novoAgendamento->accounttype = $destinatario->accounttype;
            $novoAgendamento->bank_id = $destinatario->bank_id;
            $novoAgendamento->bankbranch = $destinatario->bankbranch;
            $novoAgendamento->bankaccountdigit = $destinatario->bankaccountdigit;
            $novoAgendamento->bankaccount = $destinatario->bankaccount;

            if (taxnumberEstaDuplicado($destinatario->fromtaxnumber)) {
                $novoAgendamento->operationstatus = "ERRO_INCONSISTENTE";
                $novoAgendamento->operationresponse = "CPF/CNPJ ativo como mais de um cliente. [AGENDAMENTO: " . datetime2br(date('Y-m-d H:i:s')) . "]";
            } else {
                $novoAgendamento->operationstatus = "AGENDADO";
            }

            $novoAgendamento->save();
        } catch (\Exception $e) {
            $logError = new Logerrorautotransferschedule();
            $logError->fill([
                "fromtaxnumber" => $objCliente->taxnumber,
                "operationtype" => $operacao,
                "message" => $e->getMessage()
            ]);
            $logError->save();
        }
    }

    public function agendarD0($cliente, $dataCompletaHoje)
    {
        $this->efetuarAgendamento($cliente, $dataCompletaHoje, "D+0");
    }

    public function agendarD1($cliente, $dataCompletaHoje)
    {
        $clientSpsService = new ClientSpsService($cliente->taxnumber);
        if ($this->verificarSeExisteAgendamento($cliente, $dataCompletaHoje, "D+1")) {
            return false;
        }
        sleep(2); // Sleep para evitar Too many Requests (429)
        if ($clientSpsService->__get('saldoRealPaged') <= 0) {
            return false;
        }
        $this->efetuarAgendamento($cliente, $dataCompletaHoje, "D+1");

    }
}
